/*
 * Copyright 2019 NXP
 * All rights reserved.
 * SPDX-License-Identifier: BSD-3-Clause
 */

/*==================================================================================================
 Include Files
==================================================================================================*/

/* must be included before stdlib.h as it contains the correct define of NULL */
#include "EmbeddedTypes.h"

#include <stdlib.h>
#include <mathfp.h>
#include <math.h>

#include "fsl_device_registers.h"
#if defined(KW38Z4_SERIES) || defined(KW38A4_SERIES)
#if !defined(RADIO_IS_GEN_3P5)
#define RADIO_IS_GEN_3P5    1
#endif
#endif

/* KSDK */
#include "fsl_clock.h"
#include "fsl_tpm.h"
#include "fsl_port.h"

#if !defined(RADIO_IS_GEN_3P5)
#include "fsl_xcvr.h"

#else
#include "nxp2p4_xcvr.h"
#include "nxp_xcvr_coding_config.h"
#include "nxp_xcvr_gfsk_bt_0p5_h_0p5_config.h"

#endif

#include "fsl_os_abstraction.h"

/* ConnFWK */
#include "TimersManager.h"
#include "Flash_Adapter.h"

#include "genfsk_interface.h"
#include "genfsk_utils.h"
#include "dbg_ram_capture.h"

#include "pde.h"
#include "pde_preprocessing.h"
#include "pde_private.h"
#include "dtest_ctrl.h"

#if !defined(__IAR_SYSTEMS_ICC__)
/* Not defined in math.h on MCUX */
extern double trunc(double d);
extern double round(double d);
extern long lround(double d);
#endif /* __IAR_SYSTEMS_ICC__ */ 
    
#if !defined(RADIO_IS_GEN_3P5)
#define SIGN_EXTND_12_16(x)     ((x) | (((x) & 0x800U) ? 0xF000U : 0x0U))
#define SIGN_EXTND_5_8(x)       ((x) | (((x) & 0xE0U) ? 0xC0U : 0x0U))

#else
#define SIGN_EXTND_12_16(x)     (((int16_t)(x))/16)

#endif

#if !defined(RADIO_IS_GEN_3P5)
#define PDE_TIMER                   TPM0
#define PDE_TIMER_IRQ               TPM0_IRQn

#else
#define PDE_TIMER                   TPM2
#define PDE_TIMER_IRQ               TPM2_IRQn

#endif

#if defined(RADIO_IS_GEN_3P5)
// configure if use OSR change or DMA decimation for lower IQ sample rate
#define USE_DMA_DECIMATION          0   
#if USE_DMA_DECIMATION
    #define DMA_DECIMATION_FACTOR   DMA_DECIMATE_BY_2
#else
    #define DMA_DECIMATION_FACTOR   DMA_NO_DECIMATE
#endif
#endif

#define PDE_IS_POWER_OF_TWO_NOT_ZERO(x) ((uint32_t)((uint32_t)(x)&(uint32_t)((uint32_t)(x)-1U)) == 0U)

/*==================================================================================================
 User definitions
==================================================================================================*/
#if PDE_USE_MEM_MANAGER
#include "MemManager.h"
#define PDEMemAlloc(x)              MEM_BufferAlloc(x)
#define PDEMemFree(x)               MEM_BufferFree(x)
#else
#include <stdlib.h>
#define PDEMemAlloc(x)              malloc(x)
#define PDEMemFree(x)               free(x)
#endif  //PDE_USE_MEM_MANAGER

#if PDE_USE_FLIB
#include "FunctionLib.h"
#define PDEMemSet(x,y,z)            FLib_MemSet(x,y,z)
#define PDEMemCpy(x,y,z)            FLib_MemCpy(x,y,z)
#else
#include <string.h>
#define PDEMemSet(x,y,z)            memset(x,y,z)
#define PDEMemCpy(x,y,z)            memcpy(x,y,z)
#endif  //PDE_USE_FLIB

#define PDE_ENABLE_TX_TRIGGER       XCVR_DTEST_CTRL = XCVR_CTRL_DTEST_CTRL_DTEST_PAGE(DTEST_TX_DIG_EN_PAGE) | XCVR_CTRL_DTEST_CTRL_DTEST_EN(1)

#define PDE_ENABLE_RX_TRIGGER       XCVR_DTEST_CTRL = XCVR_CTRL_DTEST_CTRL_DTEST_PAGE(DTEST_AA_MACTH_PAGE) | XCVR_CTRL_DTEST_CTRL_DTEST_EN(1)

#define PDE_DISABLE_TRIGGER         XCVR_DTEST_CTRL = 0

#define PDE_RX_BUFFER_SIZE                          (220U)
#define PDE_MAX_ELEMENTS_PER_GFSK_TRANSACTION       (100U)
#define PDE_MAX_RETRY_COUNT                         (5U)

/* Event Flags */
#define PDE_EVT_DMA_CAPTURE_COMPLETE                (1U<<0U)
#define PDE_EVT_INTERNAL_ERROR                      (1U<<1U)
#define PDE_EVT_GFSK_PACKET_TRANSMITTED             (1U<<2U)
#define PDE_EVT_GFSK_PACKET_RECEIVED                (1U<<3U)
#define PDE_EVT_ACCESS_ADDRESS_MATCH                (1U<<4U)
#define PDE_EVT_DATA_REQUEST_COMPLETE               (1U<<5U)
#define PDE_EVT_CALIBRATE_DC                        (1U<<6U)
#define PDE_EVT_REPORT_ERROR                        (1U<<7U)

#define PDE_ACCESS_ADDRESS_UNDEFINED                (0x00000000U)

#define PDE_FREQ_SWEEP_OFFSET_HZ                    (250000U)         /* Carrier offset for 1's modulation */
#ifndef PDE_FREQ_SWEEP_MAX_NUMBER_OF_FREQUENCIES
#define PDE_FREQ_SWEEP_MAX_NUMBER_OF_FREQUENCIES    (100U)
#endif

#if !defined(RADIO_IS_GEN_3P5)
#define PDE_F0_SWITCH_TIME_TX_US                    100U
#define PDE_F0_SWITCH_TIME_RX_US                    35U
#define PDE_FN_SWITCH_TIME_US                       240U

#else
#define PDE_F0_SWITCH_TIME_TX_US                    98U    // RD: delay from TX_DIG_EN to RX
#define PDE_F0_SWITCH_TIME_RX_US                    39U    // MD: delay from AA_MATCH  to TX, increase if RD is after MD
#define PDE_FN_SWITCH_TIME_US                       1000U

#endif

#ifndef PDE_MAX_NUMBER_OF_TONE_IQ_SAMPLES
#define PDE_MAX_NUMBER_OF_TONE_IQ_SAMPLES			(32U)
#endif

#define PDE_MAX_IQ_SAMPLES                          (PDE_MAX_NUMBER_OF_TONE_IQ_SAMPLES+(PDE_FREQ_SWEEP_MAX_NUMBER_OF_FREQUENCIES-1U)*PDE_MAX_NUMBER_OF_PHASE_IQ_SAMPLES_PER_FREQ)

#define PDE_FSWEEP_TIMER_FREQ                       32000000U

#define PDE_F0_SWITCH_COUNT_TX                      (((uint32_t)PDE_F0_SWITCH_TIME_TX_US-2U)*(uint32_t)(PDE_FSWEEP_TIMER_FREQ/1000000U))
#define PDE_F0_SWITCH_COUNT_RX                      (((uint32_t)PDE_F0_SWITCH_TIME_RX_US-2U)*(uint32_t)(PDE_FSWEEP_TIMER_FREQ/1000000U))
/* Note: in F0 switch times, we need to compensate 2uS for the first switching time */
#define PDE_FN_SWITCH_COUNT                         (PDE_FN_SWITCH_TIME_US/2U)*(PDE_FSWEEP_TIMER_FREQ/1000000U)

#define PDE_TIMEOUT_DEFAULT_US                      (10000U)

#define PDE_TIMEOUT_WAIT_FOREVER                    (0U)

#define PDE_TIMEOUT_ACK_US                          (700U)

#define PDE_TIMEOUT_GLOBAL_MS                       (500U)

#define PDE_GENFSK_COMMUNICATIONS_CHANNEL           (55)
#define PDE_ENABLE_CRC_FOR_GFSK                     (1)

#define PDE_CALIBRATION_POINTS                      (3U)

typedef enum _pde_roles{
    kPdeRole_Tx_c,
    kPdeRole_Rx_c,
    kPdeRole_None_c,
}pde_roles_t;

typedef enum _pde_device_type_tag{
    kPdeDeviceType_MD_c,
    kPdeDeviceType_RD_c,
    kPdeDeviceType_None_c,
}pde_device_type_t;

typedef enum _pde_warm_states{
    kPdeWarm_Down_c,
    kPdeWarm_Up_c
}pde_warm_states_t;

typedef enum _pde_dma_capture_type{
    kPdeDmaCapture_Normal_c,
    kPdeDmaCapture_FirstRxExecution_c,
}pde_dma_capture_t;

typedef enum _pde_device_sequence_type{
    kPdeDevSeq_RoleSwapOnly_c = 0,
    kPdeDevSeq_RoleSwapFreqUpdate_c,
    kPdeDevSeq_InvalidSequence_c,
}pde_device_sequence_t;

typedef enum _pde_messaging_opcode_tag{
    kPdeMessagingOpCode_Ack_c,
    kPdeMessagingOpCode_StartTest_c,
    kPdeMessagingOpCode_StartCalibration_c,
    kPdeMessagingOpCode_StartSync_c,
    kPdeMessagingOpCode_DataRequest_c,
    kPdeMessagingOpCode_DataResponse_c,

    /* this enables 16-bit enum as the opcode is transmitted in the packet
       using two bytes */
    kPdeMessagingOpCode_Dummy_c = 0xFFFFU
}pde_messaging_opcode_t;

typedef enum _pde_data_type_tag{
    kPdeDataType_IQData_c,
    kPdeDataType_PhaseData_c,
    kPdeDataType_FreqMaskData_c,
    kPdeDataType_Count_c,
}pde_data_type_t;

typedef enum _pde_action_tag{
    kPdeAction_StartPdeSyncTx_c,
    kPdeAction_StartPdeSyncRx_c,
    kPdeAction_Critical_c = 0x80,
    kPdeAction_NoAction_c,
    kPdeAction_StartCalibrationPacketTx_c,
    kPdeAction_StartCalibrationPacketRx_c,
    kPdeAction_StartGenfskRx_c,
    kPdeAction_StartDataCapture_c,
    kPdeAction_ReportIQDataReady_c,
    kPdeAction_ReportPhaseDataReady_c,
    kPdeAction_WaitForAck_c,
    kPdeAction_ParseDataRequestOnAckSent_c,
    kPdeAction_ParseDataResponseOnAckSent_c,
    kPdeAction_ReportFreqMaskReady_c,
}pde_action_t;

typedef enum _pde_genfsk_tx_packet_type_tag{
    kPdeGenfskTxPacketType_FirstTxAttempt_c,
    kPdeGenfskTxPacketType_Retry_c,
    kPdeGenfskTxPacketType_Ack_c,
}pde_genfsk_tx_packet_type_t;

typedef enum _pde_measurement_sequence_tag{
    kPdeMeasurementSequence_Idle_c,
    kPdeMeasurementSequence_Start_c,
    kPdeMeasurementSequence_Calibration_c,
    kPdeMeasurementSequence_Sync_c,
    kPdeMeasurementSequence_DataSharing_c,
} pde_measurement_sequence_t;

typedef enum _pde_start_calibration_tag{
    kPdeStartCalibration_Rx_c,
    kPdeStartCalibration_Tx_c
}pde_start_calibration_t;

/*==================================================================================================
 Private prototypes
==================================================================================================*/
static void PDE_PeriodicTask(void* pParam);
static void PDE_ResetSystemToDefault (void);
static void PDE_IqDmaInit (void);
static void PDE_IqDmaStartCapture (uint16_t NumberOfSamples, uint32_t* pOutputBuffer, pde_dma_capture_t dmaCaptureType);
static void PDE_CalculateFineTune (uint32_t freq, uint32_t refOsc, pde_freq_calibration_t* pFrequencyRegisters);
#ifndef PDE_USE_ADVANCED_CORSE_TUNE_COMPUTATION
void PDE_CalculateCoarseTune(uint8_t GfskInstance, uint8_t * CoarseLut, uint32_t StartFreq,  uint32_t FreqStep, uint16_t NumberOfFrequencies);
#else
void PDE_CalculateCoarseTuneEnhanced(uint8_t GfskInstance, uint8_t * CoarseLut, uint32_t StartFreq,  uint32_t FreqStep, uint16_t NumberOfFrequencies);
#endif
static void PDE_FrequencySweepInit (void);
static void PDE_FrequencySweepStart(void);
static void PDE_FrequencySweepStop (void);
static inline void PDE_GetManualCalibrationValues(void);
static inline void PDE_ApplyManualCalibration(void);
static inline void PDE_ClearManualCalibration(void);
static void PDE_StartGenfskTx (pde_genfsk_tx_packet_type_t packetType, uint8_t *pData, uint16_t elementCount, uint64_t startDelayUs);
static void PDE_StartGenfskRx (uint32_t timeoutUs);
static void PDE_SendStartRequest (void);
static void PDE_DataRequest (pde_data_type_t dataType);
static void PDE_DataResponse (pde_data_type_t dataType, int16_t* pData, uint16_t offset, uint16_t dataCount);
static void PDE_StartCalibration (pde_start_calibration_t calibrationPacketAction, uint8_t calibrationChannel);
static void PDE_StartSyncTx (void);
static void PDE_StartSyncRx (void);

static void PDE_GfskAccessAddressMatchEvent (pde_action_t actionToExecute);
static void PDE_GfskPacketTransmittedEvent (pde_action_t actionToExecute);
static void PDE_GfskPacketReceivedEvent (pde_action_t actionToExecute);
static void PDE_DataRequestCompleteEvent (pde_action_t actionToExecute);
static void PDE_GenfskPacketParseRequest (uint16_t *pInPacket);
static void PDE_GenfskPacketParseResponse (uint16_t *pInPacket);
static uint8_t PDE_SetNewAction (pde_action_t newAction);

static void PDE_InitTimeoutTimer(void);
static void PDE_StartTimeoutTimer(uint32_t timeMs);
static void PDE_StopTimeoutTimer(void);
static void PDE_TimeoutCallback (void *pParam);
static void PDE_TPM_IRQHandler (void);

static void PDE_ResetRadioToDefaults(void);

/* Callbacks */
static void PDE_RxCallback (uint8_t *pBuffer, uint16_t bufferLength, uint64_t timestamp, int8_t rssi, uint8_t crcValid);
static void PDE_EventCallback (genfskEvent_t event, genfskEventStatus_t eventStatus);
#if !defined(RADIO_IS_GEN_3P5)
static void PDE_DmaCallback (struct _edma_handle *handle, void *userData, bool transferDone, uint32_t tcds);
#else
static void PDE_DmaCallback (void *userData, bool transferDone, uint16_t count);
#endif

/*==================================================================================================
 Private variables
==================================================================================================*/
static OSA_TASK_DEFINE(PDE_PeriodicTask, OSA_PRIORITY_NORMAL, 1, 1024, FALSE);
static osaTaskId_t pdePeriodicTaskId;
static osaEventId_t gPdeEventId;

static GENFSK_packet_config_t pdePacketConfig =
{
    .preambleSizeBytes = 0,
    .packetType = gGenfskFormattedPacket,
    .lengthSizeBits = 8,
    .lengthBitOrder = gGenfskLengthBitLsbFirst,
    /*sync address bytes = size + 1*/
    .syncAddrSizeBytes = 3,
    .lengthAdjBytes = 3, /*length field not including CRC so adjust by crc len*/
    .h0SizeBits = 0,
    .h1SizeBits = 0,
};

static GENFSK_radio_config_t pdeRadioConfig =
{
    .radioMode = gGenfskGfskBt0p5h0p5,
    .dataRate = gGenfskDR1Mbps
};

static GENFSK_nwk_addr_match_t pdeNetworkAddress =
{
    .nwkAddrSizeBytes = 3,
    .nwkAddrThrBits = 0,
    .nwkAddr = PDE_ACCESS_ADDRESS_UNDEFINED,
};

/*CRC configuration */
static GENFSK_crc_config_t pdeCrcConfig =
{
    .crcEnable = gGenfskCrcEnable,
    .crcSize = 3,
    .crcStartByte = 4,
    .crcRefIn = gGenfskCrcInputNoRef,
    .crcRefOut = gGenfskCrcOutputNoRef,
    .crcByteOrder = gGenfskCrcLSByteFirst,
    .crcSeed = 0x00555555,
    .crcPoly = 0x0000065B,
    .crcXorOut = 0
};

/* Whitening configuration */
static GENFSK_whitener_config_t pdeWhitenerConfig =
{
    .whitenEnable = gGenfskWhitenEnable,
    .whitenStart = gWhitenStartWhiteningAtH0,
    .whitenEnd = gWhitenEndAtEndOfCrc,
    .whitenB4Crc = gCrcB4Whiten,
    .whitenPolyType = gFibonnaciPolyType,
    .whitenRefIn = gGenfskWhitenInputNoRef,
    .whitenPayloadReinit = gGenfskWhitenNoPayloadReinit,
    .whitenSize = 7,
    .whitenInit = 0x53,
    .whitenPoly = 0x11,
    .whitenSizeThr = 0x0,
    .manchesterEn = gGenfskManchesterDisable,
    .manchesterStart = gGenfskManchesterStartAtPayload,
    .manchesterInv = gGenfskManchesterNoInv,
};

static uint8_t gPdeGfskInstanceId = gGENFSK_InvalidIdx_c;

#if ((PDE_RX_BUFFER_SIZE)%2) != 0 
#error PDE_RX_BUFFER_SIZE must be even
#endif
static uint16_t pPdeRxBuffer16[(PDE_RX_BUFFER_SIZE)/2U];

static uint16_t gSamplesInBuffer = 0;
static uint16_t gRequestedSamples = 0;

/* the following buffer allocation must be 32-bit aligned */
static int16_t *gIqSamplesBuffer;
/* DMA functions need a pointer to a 32-bit aligned buffer */
#define gIqSamplesBuffer32  ((uint32_t*)(void*)gIqSamplesBuffer)

static int16_t  gPdeLocalPhaseBuffer[PDE_FREQ_SWEEP_MAX_NUMBER_OF_FREQUENCIES];
static int16_t  gPdeRemotePhaseBuffer[PDE_FREQ_SWEEP_MAX_NUMBER_OF_FREQUENCIES];

/* Global system configuration */
static pde_system_config_t gPdeSysCfg;

/*! Current running capture */
static pde_capture_config_t gPdeCaptureCfg;

static uint32_t gFirstFsweepCounts;
static uint32_t gNextFsweepCounts;

static pde_roles_t gPdeCurrentRole = kPdeRole_None_c;
static pde_device_type_t gPdeCurrentDeviceType = kPdeDeviceType_None_c;

static pde_freq_calibration_t gPdeFineTuneConfigTable[2][PDE_FREQ_SWEEP_MAX_NUMBER_OF_FREQUENCIES]; //Radio Configurations for Frequency Sweep
static uint8_t gPdeCoarseTuneConfigTable[PDE_FREQ_SWEEP_MAX_NUMBER_OF_FREQUENCIES];

static int16_t *gPdePeerIQDataBuffer = NULL;
static int16_t *pPdeCombinedPhaseBuffer = NULL;
static uint32_t * pPdeFreqMask = NULL;

static bool_t gPdeCalibrationDone = false;
static uint8_t gPdeCalibrationIndex;
static uint8_t gPdeCalibrationChannelIndex;
static uint8_t gPdeCalibrationChannel[PDE_CALIBRATION_POINTS];
static pde_calibration_t gPdeCalibration[PDE_CALIBRATION_POINTS];
static uint8_t aPdeCalibrationSyncPacket[19];

static struct pde_remaining_data_tag{
    int16_t*        dataPointer;
    uint16_t        originalElementCount;
    uint16_t        remainingElements;
    pde_data_type_t dataType;
    bool_t          isDataPending;
}gPdeRemainingData;

static pde_action_t gPdeActionOnEvent = kPdeAction_NoAction_c;
static pde_action_t gPdeLastActionExecuted = kPdeAction_NoAction_c;

static tmrTimerID_t gPdeTimeoutTimerId;

static pde_error_t gPdeReportedError;

static int16_t aPdeLastMdPreprocessedPhase[PDE_FREQ_SWEEP_MAX_NUMBER_OF_FREQUENCIES];
static int16_t aPdeLastRdPreprocessedPhase[PDE_FREQ_SWEEP_MAX_NUMBER_OF_FREQUENCIES];

static uint16_t gPdeRemainingDmaCaptures;

static uint16_t gPdeLocalSequenceNumber;

static uint32_t FreqMaskLocal[(PDE_FREQ_SWEEP_MAX_NUMBER_OF_FREQUENCIES / 32) + 1];

static uint32_t FreqMaskPeer[(PDE_FREQ_SWEEP_MAX_NUMBER_OF_FREQUENCIES / 32) + 1];

static pde_measurement_sequence_t gPdeMeasurementSequenceStage = kPdeMeasurementSequence_Idle_c;

/*==================================================================================================
 Public APIs
==================================================================================================*/
/* Wrapper to preprocessing module */
void PDE_SetPreprocessingOptions(bool_t frequencyMaskEnable, bool_t phaseMaskEnable){
    PDE_PreprocessingEnableFrequencyMask((bool)frequencyMaskEnable);
    PDE_PreprocessingEnablePhaseMask((bool)phaseMaskEnable);
}

/*
 * Initialize to default values.
 * This initializer is required for backward compatibility, to
 * allow pde_system_config_t extension in the future.
 */
void PDE_InitSysConfig(pde_system_config_t *pdeSysCfg)
{
    /* set to default values */
    pdeSysCfg->freqSweepIndentHz = 500000U;
    pdeSysCfg->freqSweepNb = PDE_FREQ_SWEEP_MAX_NUMBER_OF_FREQUENCIES;
    pdeSysCfg->freqSweepStartHz = 2405000000U;
    pdeSysCfg->localAccessAddress = PDE_ACCESS_ADDRESS_UNDEFINED;
    pdeSysCfg->peerAccessAddress = PDE_ACCESS_ADDRESS_UNDEFINED;
    pdeSysCfg->pdeEventCallback = NULL;
    pdeSysCfg->nbOfSamplesPerFrequency = PDE_NUMBER_OF_PHASE_IQ_SAMPLES_PER_FREQ;
}

/*
 * Initialize to default values.
 * This initializer is required for backward compatibility, to
 * allow pde_system_config_t extension in the future.
 */
void PDE_InitCaptureConfig(pde_capture_config_t *pdeCaptureCfg)
{
    pdeCaptureCfg->reportFlags = 0;
}

pde_status_t PDE_Init (pde_system_config_t *pdeSysCfg)
{
    static bool_t gPdeInitialized = FALSE;
    pde_status_t pde_status = kPdeStatus_Ok_c;

    do {
        genfskStatus_t status = gGenfskFail_c;

        /* Sanity check input parameters */
        if ((pdeSysCfg->freqSweepNb > PDE_FREQ_SWEEP_MAX_NUMBER_OF_FREQUENCIES) ||
                (pdeSysCfg->freqSweepStartHz < 2400000000U) ||
                ((pdeSysCfg->freqSweepStartHz + pdeSysCfg->freqSweepNb*pdeSysCfg->freqSweepIndentHz)>2480000000U) ||
                (pdeSysCfg->pdeEventCallback == NULL) ||
                (pdeSysCfg->localAccessAddress == PDE_ACCESS_ADDRESS_UNDEFINED) ||
                (pdeSysCfg->peerAccessAddress == PDE_ACCESS_ADDRESS_UNDEFINED)  ||
                (!PDE_IS_POWER_OF_TWO_NOT_ZERO(pdeSysCfg->nbOfSamplesPerFrequency))) {
            pde_status = kPdeStatus_InvalidInputParams_c;
            break;
        }

        if(!gPdeInitialized){

            //Create application task
            pdePeriodicTaskId = OSA_TaskCreate(OSA_TASK(PDE_PeriodicTask), NULL);
            assert(pdePeriodicTaskId != NULL);
            NOT_USED(pdePeriodicTaskId);

            //Create application event
            gPdeEventId = OSA_EventCreate(true);
            assert(gPdeEventId != NULL);

            /* Initialize Timeout Timer */
            PDE_InitTimeoutTimer();

            gPdeInitialized = TRUE;
        }

        /* Save system config */
        gPdeSysCfg = *pdeSysCfg;

        /* Store local AA */
        pdeNetworkAddress.nwkAddr = gPdeSysCfg.localAccessAddress;

        /* Reset Radio */
        XCVR_Reset();

        /* Allocate IQ buffer */
        if (gIqSamplesBuffer == NULL)
        {
            gIqSamplesBuffer = (int16_t *)PDEMemAlloc(PDE_MAX_IQ_SAMPLES*2U*sizeof(int16_t));
            if ((gIqSamplesBuffer == NULL) )
            {
                pde_status = kPdeStatus_OutOfMemory_c;
                break;
            }
        }

        /* Setup GENFSK instance */
        if (gPdeGfskInstanceId == gGENFSK_InvalidIdx_c) {

            /* Initialize GENFSK */
            status = GENFSK_Init();
            assert((status == gGenfskSuccess_c) || (status == gGenfskAlreadyInit_c));
            NOT_USED(status);

            status = GENFSK_AllocInstance(&gPdeGfskInstanceId, NULL, NULL, NULL);
            assert(status == gGenfskSuccess_c);
            NOT_USED(status);

            /* Register callback */
            status = GENFSK_RegisterCallbacks(gPdeGfskInstanceId, PDE_RxCallback, PDE_EventCallback);
            assert(status == gGenfskSuccess_c);
            NOT_USED(status);
        }

        /* Initialize Frequency Sweep */
        PDE_FrequencySweepInit();

        /* Reset radio to defaults */
        PDE_ResetRadioToDefaults();

        /* Initialize DMA for IQ capture */
        PDE_IqDmaInit();

        /* Obtain the PDE calibration channels */
        gPdeCalibrationChannel[0] = (uint8_t)((pdeSysCfg->freqSweepStartHz - 2360000000U)/1000000U);
        gPdeCalibrationChannel[2] = (uint8_t)(((pdeSysCfg->freqSweepStartHz + (pdeSysCfg->freqSweepIndentHz*pdeSysCfg->freqSweepNb))-2360000000U)/1000000U);
        gPdeCalibrationChannel[1] = (uint8_t)(gPdeCalibrationChannel[0]+((gPdeCalibrationChannel[2]-gPdeCalibrationChannel[0])/2U));

        /* Initialize debug ports */
        PORT_SetPinMux(PORTA, 16U, kPORT_MuxAsGpio);    //D0
        PORT_SetPinMux(PORTC, 19U, kPORT_MuxAsGpio);    //D1
        PORT_SetPinMux(PORTC, 3U, kPORT_MuxAsGpio);     //D2

        GPIOA->PDDR |= ((uint32_t)1U<<16U);
        GPIOA->PCOR |= ((uint32_t)1U<<16U);
        GPIOC->PDDR |= ((uint32_t)1U<<3U | (uint32_t)1<<19U);
        GPIOC->PCOR |= ((uint32_t)1U<<3U | (uint32_t)1<<19U);

        /* Initialize RX Ready LED */
        PORT_SetPinMux(PORTC, 1U, kPORT_MuxAsGpio);
        GPIOC->PDDR |= (1U<<1U);
        GPIOC->PCOR |= (1U<<1U);

        /* Generate the packet used for sync and calibration */
        GENFSK_packet_t pdePacket = {
                .header.h0Field = 0,
                .header.h1Field = 0,
                .header.lengthField = 14U, /* payload len */
                .addr = gPdeSysCfg.peerAccessAddress,
                .payload = NULL /* Use no-copy option for  payload */
        };

        /* Fill packet buffer with ones */
        PDEMemSet(aPdeCalibrationSyncPacket, 0xFFU, sizeof(aPdeCalibrationSyncPacket));

        /* Write packet header in aPdeCalibrationSyncPacket */
        status = GENFSK_PacketToByteArray(gPdeGfskInstanceId, &pdePacket, &aPdeCalibrationSyncPacket[0]);
        assert(status == gGenfskSuccess_c);
        NOT_USED(status);

        PDE_PreprocessingInit();
#if PDE_DEBUG_ADD_CARRIER_OFFSET
        PDE_PreprocessingEnableFrequencyMask(false);
        PDE_PreprocessingEnablePhaseMask(false);
#else
        PDE_PreprocessingEnableFrequencyMask(true);
        PDE_PreprocessingEnablePhaseMask(true);
#endif

#if PDE_DEBUG_GFSK_PACKET_TIMING
        PDE_EnablePacketTimingDebug();
#endif //PDE_DEBUG_GFSK_PACKET_TIMING

    } while (FALSE);

    return pde_status;
}

/* Stop PDE activity, reset system to defaults */
void PDE_Stop(void)
{
    /* Abort any pending transaction */
    GENFSK_AbortAll();

    /* Free IQ buffers */   
    (void)PDEMemFree(gIqSamplesBuffer);
    if (gPdePeerIQDataBuffer != NULL) {
        (void)PDEMemFree(gPdePeerIQDataBuffer);
    }

    gIqSamplesBuffer = NULL;
    gPdePeerIQDataBuffer = NULL;

    (void)GENFSK_FreeInstance(gPdeGfskInstanceId);
    gPdeGfskInstanceId = gGENFSK_InvalidIdx_c;

    /* Cleanup TPM interrupts */
    TPM_DisableInterrupts(PDE_TIMER, (uint32_t)kTPM_TimeOverflowInterruptEnable);

    PDE_ResetSystemToDefault();

    gPdeCurrentDeviceType = kPdeDeviceType_None_c;
}

/* Start listening forever */
void PDE_StartListening(void)
{
    gPdeCurrentDeviceType = kPdeDeviceType_RD_c;

    /* Start wait mode */
    PDE_StartGenfskRx(PDE_TIMEOUT_WAIT_FOREVER);
}

static void PDE_PeriodicTask_EventDmaComplete(void)
{
    /* Stop timeout Timer */
    PDE_StopTimeoutTimer();

    /* LED indicates end of RX */
    GPIOC->PCOR |= (1U<<1U);

    /* Reset Last action */
    gPdeLastActionExecuted = kPdeAction_NoAction_c;

    gPdeMeasurementSequenceStage = kPdeMeasurementSequence_DataSharing_c;

    /* Reset radio to defaults */
    PDE_ResetRadioToDefaults();

    /* Start data preprocessing */
#if !defined(RADIO_IS_GEN_3P5)
    // Data is originally 12-bits signed. Extend to 16-bit signed
    sign_extend_12_16((uint16_t*)gIqSamplesBuffer, gSamplesInBuffer*2);
#else
    // Divide the samples by 16 (shift by 4) as they are 12-bit and MSbit aligned
    for (uint32_t i = 0; i<(uint32_t)gSamplesInBuffer*2U; i++) {
        gIqSamplesBuffer[i] /= 16;
    }
#endif

    // Execute preprocessing function
    PDE_PreprocessingSamples(&gIqSamplesBuffer[PDE_MAX_NUMBER_OF_TONE_IQ_SAMPLES*2U], gPdeSysCfg.nbOfSamplesPerFrequency, \
        gPdeSysCfg.freqSweepNb-1U, &gPdeLocalPhaseBuffer[0], &FreqMaskLocal[0]);

    if(gPdeCurrentDeviceType == kPdeDeviceType_MD_c){
        /* Wait 5mS for RD to be ready*/
        PDE_TimeDelayUs(5000);
        //Request the phase samples to peer device
        PDE_DataRequest(kPdeDataType_PhaseData_c);
    }
    else{
        PDE_ResetSystemToDefault();
        /* Start GFSK RX */
        PDE_StartGenfskRx(PDE_TIMEOUT_WAIT_FOREVER);
    }
}

static void PDE_PeriodicTask_EventCalibrateDc(void)
{
    /* Average the tone samples and correct DC with them */
    int32_t iAccumulator = 0, qAccumulator = 0;
    int16_t iSample, qSample;
    int16_t dcocI, dcocQ;

    for(uint16_t counter = 0; counter<PDE_MAX_NUMBER_OF_TONE_IQ_SAMPLES; counter++){
        /* Get IQ samples */
        iSample = gIqSamplesBuffer[(counter*2U)];
        qSample = gIqSamplesBuffer[((counter*2U)+1U)];

        /* Extend sign */
        iSample = SIGN_EXTND_12_16(iSample);
        qSample = SIGN_EXTND_12_16(qSample);

        /* Accumulate */
        iAccumulator += iSample;
        qAccumulator += qSample;
    }

    /* Average I and Q */
    iAccumulator /= (int32_t)PDE_MAX_NUMBER_OF_TONE_IQ_SAMPLES;
    qAccumulator /= (int32_t)PDE_MAX_NUMBER_OF_TONE_IQ_SAMPLES;

    /* Get the current DCOC Calibration values for I and Q */
    dcocI = (int16_t)((uint16_t)((XCVR_RX_DIG->DCOC_DIG_MAN & XCVR_RX_DIG_DCOC_DIG_MAN_DIG_DCOC_INIT_I_MASK)>>XCVR_RX_DIG_DCOC_DIG_MAN_DIG_DCOC_INIT_I_SHIFT));
    dcocQ = (int16_t)((uint16_t)((XCVR_RX_DIG->DCOC_DIG_MAN & XCVR_RX_DIG_DCOC_DIG_MAN_DIG_DCOC_INIT_Q_MASK)>>XCVR_RX_DIG_DCOC_DIG_MAN_DIG_DCOC_INIT_Q_SHIFT));

    /* Extend sign */
    dcocI = SIGN_EXTND_12_16(dcocI);
    dcocQ = SIGN_EXTND_12_16(dcocQ);

    /* Add average values */
    dcocI += (int16_t)iAccumulator;
    dcocQ += (int16_t)qAccumulator;

    /* Apply new calibration values */
    XCVR_RX_DIG->DCOC_DIG_MAN = XCVR_RX_DIG_DCOC_DIG_MAN_DIG_DCOC_INIT_Q(dcocQ) | XCVR_RX_DIG_DCOC_DIG_MAN_DIG_DCOC_INIT_I(dcocI);
}

static void PDE_PeriodicTask(void* pParam){
    osaEventFlags_t pdeFlags;

    /* Check for events */
    (void)OSA_EventWait(gPdeEventId, osaEventFlagsAll_c, false, osaWaitForever_c, &pdeFlags);

    if(pdeFlags != 0U){
        /* Backup action to execute */
        gPdeLastActionExecuted = gPdeActionOnEvent;
        /* Clear currect action */
        gPdeActionOnEvent = kPdeAction_NoAction_c;
    }

    if( (pdeFlags & PDE_EVT_DMA_CAPTURE_COMPLETE) != 0U ){
        PDE_PeriodicTask_EventDmaComplete();
    }

    if( (pdeFlags & PDE_EVT_INTERNAL_ERROR) != 0U ){
        /* Check if the error requires retry */
        switch(gPdeReportedError){
          case kPdeError_CrcInvalid_c:
          case kPdeError_UnknownOpCode_c:
          case kPdeError_GfsxRxError_c:
            {
                if(gPdeLastActionExecuted == kPdeAction_WaitForAck_c){
                    /* ACK reception failed, retry the last packet */
                    PDE_StartGenfskTx(kPdeGenfskTxPacketType_Retry_c, NULL, 0, 0);
                }
                else{
                    if(gPdeMeasurementSequenceStage != kPdeMeasurementSequence_Idle_c){
                        /* Start RX again. Peer device might retry the previous packet */
                        PDE_StartGenfskRx(PDE_TIMEOUT_DEFAULT_US);
                    }
                    else{
                        /* Some garbage was received. Start RX wait forever again */
                        PDE_StartGenfskRx(PDE_TIMEOUT_WAIT_FOREVER);
                    }
                }
            }
            break;

          case kPdeError_Timeout_c:
            {
                if(gPdeLastActionExecuted == kPdeAction_WaitForAck_c){
                    /* Timed out waiting for ACK, retry the last packet */
                    PDE_StartGenfskTx(kPdeGenfskTxPacketType_Retry_c, NULL, 0, 0);
                }
                else
                {
                    switch(gPdeMeasurementSequenceStage){
                      case kPdeMeasurementSequence_Start_c:{
                          gPdeReportedError = kPdeError_StartPacketTimeout_c;
                      }
                      break;

                      case kPdeMeasurementSequence_Calibration_c:{
                          gPdeReportedError = kPdeError_CalibrationPacketTimeout_c;
                      }
                      break;

                      case kPdeMeasurementSequence_Sync_c:{
                          gPdeReportedError = kPdeError_SyncPacketTimeout_c;
                      }
                      break;

                      case kPdeMeasurementSequence_DataSharing_c:{
                          gPdeReportedError = kPdeError_DataSharingTimeout_c;
                      }
                      break;

                      default:{
                          gPdeReportedError = kPdeError_Timeout_c;
                      }
                      break;
                    }
                    /* Report the timeout error to the application */
                    (void)OSA_EventSet(gPdeEventId, PDE_EVT_REPORT_ERROR);
                }
                break;
            }
          default:
            {
                /* These are unrecoverable errors. Report to the application */
                (void)OSA_EventSet(gPdeEventId, PDE_EVT_REPORT_ERROR);
            }
            break;
        }
    }


    if( (pdeFlags & PDE_EVT_REPORT_ERROR) != 0U ){
        gPdeSysCfg.pdeEventCallback(kPdeEvent_ErrorOccurred_c, gPdeReportedError);

        /* Stop timeout Timer */
        PDE_StopTimeoutTimer();

        /* Restore the normal CRC configuration */
#if PDE_ENABLE_CRC_FOR_GFSK
        pdeCrcConfig.crcEnable = gGenfskCrcEnable;
#else
        pdeCrcConfig.crcEnable = gGenfskCrcDisable;
#endif
        (void)GENFSK_SetCrcConfig(gPdeGfskInstanceId, &pdeCrcConfig);

        /* Reset Last action */
        gPdeLastActionExecuted = kPdeAction_NoAction_c;

        /* Reset to default */
        PDE_ResetSystemToDefault();

        /* Reset radio to defaults */
        PDE_ResetRadioToDefaults();

        if (gPdeCurrentDeviceType == kPdeDeviceType_RD_c) {
            /* Start wait mode */
            PDE_StartGenfskRx(PDE_TIMEOUT_WAIT_FOREVER);
        }
    }

    if( (pdeFlags & PDE_EVT_GFSK_PACKET_RECEIVED) != 0U ){
        PDE_GfskPacketReceivedEvent(gPdeLastActionExecuted);
    }

    if( (pdeFlags & PDE_EVT_GFSK_PACKET_TRANSMITTED) != 0U ){
        PDE_GfskPacketTransmittedEvent(gPdeLastActionExecuted);
    }

    if( (pdeFlags & PDE_EVT_ACCESS_ADDRESS_MATCH) != 0U ){
        PDE_GfskAccessAddressMatchEvent(gPdeLastActionExecuted);
    }

    if( (pdeFlags & PDE_EVT_DATA_REQUEST_COMPLETE) != 0U ){
        PDE_DataRequestCompleteEvent(gPdeLastActionExecuted);
    }

    if( (pdeFlags & PDE_EVT_CALIBRATE_DC) != 0U ){
        PDE_PeriodicTask_EventCalibrateDc();
    }
}

pde_status_t PDE_StartDataCapture (pde_capture_config_t *pCaptureCfg, int16_t *pOutputPhaseBuffer, uint32_t *pFreqMask)
{
    pde_status_t status = kPdeStatus_Ok_c;

    if ((pCaptureCfg == NULL) || (pOutputPhaseBuffer == NULL) || (pFreqMask == NULL)) {
        status = kPdeStatus_InvalidInputParams_c;
    } else {
        if ((pCaptureCfg->reportFlags & PDE_REPORT_RAW_IQ) != 0U) {
            if (NULL == gPdePeerIQDataBuffer) {
                gPdePeerIQDataBuffer = (int16_t *)PDEMemAlloc(PDE_MAX_IQ_SAMPLES*2U*sizeof(int16_t));
            }
            if (gPdePeerIQDataBuffer == NULL) {
                status = kPdeStatus_OutOfMemory_c;
            }
        }
        if (status == kPdeStatus_Ok_c) {
            if ((0U == PDE_SetNewAction(kPdeAction_StartDataCapture_c)) && (gPdeGfskInstanceId != gGENFSK_InvalidIdx_c)){

                gPdeCaptureCfg = *pCaptureCfg;
                
                /* Init state machine */
                gPdeActionOnEvent = kPdeAction_NoAction_c;
                
                /* Start Timeout Timer */
                PDE_StartTimeoutTimer(PDE_TIMEOUT_GLOBAL_MS);
                
                /* Indicate Start with LED */
                GPIOC->PSOR |= (1U<<1U);
                
                gPdeCurrentDeviceType = kPdeDeviceType_MD_c;
                gPdeCurrentRole = kPdeRole_Rx_c;
                
                /* Capture the samples to take. It must be a multiple of the DMA samples per capture */
                gPdeRemainingDmaCaptures = gPdeSysCfg.freqSweepNb;
                gRequestedSamples = PDE_MAX_NUMBER_OF_TONE_IQ_SAMPLES + 
                    gPdeSysCfg.nbOfSamplesPerFrequency * (gPdeRemainingDmaCaptures-1U);
                
                /* Save the output Phase Buffer */
                pPdeCombinedPhaseBuffer = pOutputPhaseBuffer;
                pPdeFreqMask = pFreqMask;
                
                /* Reset the Calibration Index */
                gPdeCalibrationIndex = 0;
                gPdeCalibrationChannelIndex = 0;
                gPdeCalibrationDone = false;
                
                /* Send Start Request to Peer */
                PDE_SendStartRequest();
            } else {
                status = kPdeStatus_NotReady_c;
            }
        }
    }
    return status;
}

pde_error_t PDE_FlushLocalIqSamples (int16_t* iSamples, int16_t* qSamples){
  
    pde_error_t status = kPdeError_Ok_c;
    int16_t* pSamplePointer = (int16_t*)gIqSamplesBuffer;

    if(gSamplesInBuffer == 0U){
        /* nothing to do */
        status = kPdeError_SamplesBufferAlreadyFlushed_c;
    }
    else{

        /*Store I and Q samples in different buffers*/
        while (gSamplesInBuffer != 0U) {
                *iSamples++ = *pSamplePointer++;
                *qSamples++ = *pSamplePointer++;
                gSamplesInBuffer--;
        }

        /* Clear IQ Samples buffer */
        for(uint32_t index = 0; index < (PDE_MAX_IQ_SAMPLES*2U); index++){
            gIqSamplesBuffer[index]=0;
        }
    }

    return status;
}

pde_error_t PDE_FlushPeerIqSamples (int16_t* iSamples, int16_t* qSamples){
    /* reset remote sample buffers */
    for(uint16_t index = 0; index<PDE_MAX_IQ_SAMPLES; index++){
        /* I samples */
        iSamples[index] = gPdePeerIQDataBuffer[index*2U];
        gPdePeerIQDataBuffer[index*2U] = 0;
        /* Q samples */
        qSamples[index] = gPdePeerIQDataBuffer[index*2U+1U];
        gPdePeerIQDataBuffer[index*2U+1U] = 0;
    }

    return kPdeError_Ok_c;
}

void PDE_FlushPeerPreprocessedPhase (int16_t *pOutputBuffer){
    /* get remote phase */
    PDEMemCpy(pOutputBuffer, &aPdeLastRdPreprocessedPhase[0], sizeof(aPdeLastRdPreprocessedPhase));
}

void PDE_FlushLocalPreprocessedPhase (int16_t *pOutputBuffer){
    /* get local phase */
    PDEMemCpy(pOutputBuffer, &aPdeLastMdPreprocessedPhase[0], sizeof(aPdeLastMdPreprocessedPhase));
}

pde_error_t PDE_CalculateIqPhase(uint16_t iqSampleNumber, int16_t* iSamples, int16_t* qSamples, int16_t* outputBuffer){
  
    /* some checks */
    assert(iSamples != NULL);
    assert(qSamples != NULL);
    assert(outputBuffer != NULL);

    uint16_t i;
    int16_t * out_ptr = outputBuffer;
    int16_t result;

    for (i=0;i<iqSampleNumber;i++)
    {
        /* compute the phase */
        result = atan2fp(*qSamples++,*iSamples++);
        *out_ptr++ = result;
    }

    return kPdeError_Ok_c;
}

void PDE_GetMeasurementSettings(pde_measure_settings_t *settings)
{
    /* get settings */
    settings->samplesCaptured = gRequestedSamples;
    settings->agcIndex = gPdeCalibration[gPdeCalibrationIndex].agcIndex;
}

void PDE_GetPllSettings(pde_freq_calibration_t** PllSettings, uint8_t** CoarseTune, uint16_t * NumberFrequencies)
{
    /* get settings */
    *PllSettings = (pde_freq_calibration_t*)&gPdeFineTuneConfigTable[0][0];

    *CoarseTune = (uint8_t*)&gPdeCoarseTuneConfigTable[0];

    *NumberFrequencies = gPdeSysCfg.freqSweepNb;
}

/*==================================================================================================
 Private APIs
==================================================================================================*/
static void PDE_ResetSystemToDefault (void){

    /* Reset global variables */
    gPdeCurrentRole = kPdeRole_None_c;
    gPdeActionOnEvent = kPdeAction_NoAction_c;
    gPdeMeasurementSequenceStage = kPdeMeasurementSequence_Idle_c;
    
    /* Reset transceiver DMA to the power on state */
    PDE_XcvrDmaReset();

    /* Reset TSM Overrides */
    PDE_TsmOverrideRestore();

    /* Reset frequency sweep engine */
    PDE_FrequencySweepStop();

#if !defined(RADIO_IS_GEN_3P5) || USE_DMA_DECIMATION==0
    /* Reset OSR */
    PDE_ResetOsr();
#endif

    /* Release manual DC calibration */
    PDE_ClearManualCalibration();

    /* Disable AA Match event */
    (void)GENFSK_SetEventMask(gPdeGfskInstanceId, gGenfskTxEvent | gGenfskRxEvent | gGenfskWakeEvent);

#if !defined(RADIO_IS_GEN_3P5)
    /* Disable DTEST interrupt that triggers PDE_TIMER */
    PORT_SetPinInterruptConfig(PORTB, 2U, kPORT_InterruptOrDMADisabled);
#endif

    /* Turn off status LED */
    GPIOC->PCOR |= (1U<<1U);
}

static void PDE_IqDmaInit (void){
    /* init DMA driver */
    PDE_XcvrDmaInit(PDE_DmaCallback);
}

static void PDE_IqDmaStartCapture (uint16_t NumberOfSamples, uint32_t* pOutputBuffer, pde_dma_capture_t dmaCaptureType){

    /* Make sure the number of requested samples is within the limits */
    assert(NumberOfSamples <= PDE_MAX_IQ_SAMPLES);

    /* Make sure we still need to capture samples */
    assert(gPdeRemainingDmaCaptures != 0U);

#if defined(RADIO_IS_GEN_3P5)
    PDE_XcvrDmaInit(PDE_DmaCallback);
#endif

    /* Enable DMA */
    if (dmaCaptureType == kPdeDmaCapture_FirstRxExecution_c) {
        /* Configure Start Trigger: Access Address Match */
#if defined(RADIO_IS_GEN_3P5)
        /* remove the length and opcode (24us) plus data path delay (4us minimum) */
        uint16_t delay_us = (gPdeCurrentDeviceType == kPdeDeviceType_MD_c) ? 30U:0U;
        PDE_XcvrDmaStart(START_DMA_ON_FSK_AA_MATCH, pOutputBuffer, NumberOfSamples, DMA_DECIMATION_FACTOR, delay_us);
#else
        // KW36 DMA does not support delay feature
        PDE_XcvrDmaStart(START_DMA_ON_FSK_AA_MATCH, pOutputBuffer, NumberOfSamples);
#endif
    }
    else {
        /* Configure Start Trigger: No trigger (start immediately) */
#if defined(RADIO_IS_GEN_3P5)
        PDE_XcvrDmaStart(NO_DMA_START_TRIG, pOutputBuffer, NumberOfSamples, DMA_DECIMATION_FACTOR, 0);
#else
        PDE_XcvrDmaStart(NO_DMA_START_TRIG, pOutputBuffer, NumberOfSamples);
#endif
    }
    /* Report that capture has been started */
    gPdeRemainingDmaCaptures--;
}

static void PDE_CalculateFineTune (uint32_t freq, uint32_t refOsc, pde_freq_calibration_t* pFrequencyRegisters)
{
    double integer_used_in_Hz,
           integer_used_in_LSB,
           numerator_fraction,
           numerator_in_Hz,
           numerator_in_LSB,
           numerator_unrounded,
           real_int_and_fraction,
           real_fraction,
           requested_freq_in_LSB,
           sdm_lsb;
    uint32_t integer_truncated,
             integer_to_use;
    int32_t numerator_rounded;
    const uint32_t denominator = 0x08000000U - 1U; /* Configure the PLL Denominator to the max for best resolution */

    /* Write the Low Port Denomintor value */
    pFrequencyRegisters->denominator = denominator;

    /* Calculate the Low Port values */
    sdm_lsb = (double)refOsc / ((double)denominator / 2.0);
    real_int_and_fraction = (double)freq / ((double)refOsc * 2.0);

    integer_truncated = (uint32_t) trunc(real_int_and_fraction);

    real_fraction = real_int_and_fraction - (double)integer_truncated;

    if (real_fraction > 0.5)
    {
        /* round up */
        integer_to_use = integer_truncated + 1U;
    }
    else
    {
        integer_to_use = integer_truncated;
    }

    pFrequencyRegisters->integerToUse = integer_to_use;

    numerator_fraction = real_int_and_fraction - (double)integer_to_use;

    integer_used_in_Hz  = (double)((uint32_t)(integer_to_use * refOsc * 2U));
    integer_used_in_LSB = integer_used_in_Hz / sdm_lsb;

    numerator_in_Hz  = numerator_fraction * (double)((uint32_t)(refOsc * 2U));
    numerator_in_LSB = numerator_in_Hz    / sdm_lsb;

    requested_freq_in_LSB = integer_used_in_LSB + numerator_in_LSB;
    numerator_unrounded = (requested_freq_in_LSB - integer_used_in_LSB);
    numerator_rounded = (int32_t)round(numerator_unrounded);

    pFrequencyRegisters->numeratorRounded = (uint32_t)numerator_rounded;
}

/*
 * Coarse tune requires some calibration (not just computation)
 */
#ifndef PDE_USE_ADVANCED_CORSE_TUNE_COMPUTATION
void PDE_CalculateCoarseTune(uint8_t GfskInstance, uint8_t *CoarseLut, uint32_t StartFreq,
                             uint32_t FreqStep, uint16_t NumberOfFrequencies)
{
    for(uint8_t counter = 0; counter<NumberOfFrequencies; counter++){
        /* set the frequency value */
        (void)GENFSK_OverrideFrequency(StartFreq+(FreqStep*counter));

        PDE_AutoTxWarmUp();
        CoarseLut[counter] = PDE_XCVR_GET_CTUNE_SELECTED();
        PDE_AutoTxWarmDown();
    }

    /* Release PLL Manual */
    PDE_PllManualControlRelease();
}

#else /* PDE_USE_ADVANCED_CORSE_TUNE_COMPUTATION */
static void PDE_CalculateCoarseTuneEnhanced(uint8_t GfskInstance, uint8_t *CoarseLut, uint32_t StartFreq,
                                     uint32_t FreqStep, uint16_t NumberOfFrequencies)
{
    float CoarseTune;
    float CtSlope;
    float CtInit;
    float FreqDifference;
    uint32_t EndFrequency;
    uint32_t NewFreqCalc;
    uint16_t FreqOffset = 0;
    uint8_t CtMin;
    uint8_t BestDiffMin;
    uint8_t CtMax;
    uint8_t CurrentChannel;

    /* To calculate coarse tune values we'll need the following sequence */
    /*  1.	Measure the CT values as determined by calibration at minimum and
            maximum frequency, say, CTmin and CTmax. Also note down the
            best-diff at Fmin(from CTUNE_RES. CTUNE_BEST_DIFF)
        2.	Compute the CT-slope as CT-slope = (CTmax- CTmin)/(Fmax-Fmin)
        3.	Define the CT0=CTmin + CTUNE_RES. CTUNE_BEST_DIFF*CT-slope
        4.	Calculate CTune for all frequencies using the formula
            a.	ROUND(CT0+(F-Fmin)* CT-slope)                                    */

    if(CoarseLut != NULL)
    {
        /* back up current channel */
        CurrentChannel = GENFSK_GetChannelNumber(GfskInstance);

        EndFrequency = StartFreq + (FreqStep * NumberOfFrequencies);

        /* Calibrate Coarse */
        /* Override to the Min frequency */
        (void)XCVR_OverrideFrequency(StartFreq);

        PDE_AutoTxWarmUp();

        /* Get Coarse Tune Min (CtMin) and Best Diff Min (BestDiffMin) */
        CtMin = PDE_XCVR_GET_CTUNE_SELECTED();
        BestDiffMin = PDE_XCVR_GET_CTUNE_BEST_DIFF();

        PDE_AutoTxWarmDown();

        /* Override to the Max frequency */
        (void)XCVR_OverrideFrequency(EndFrequency);

        PDE_AutoTxWarmUp();

        /* Get Coarse Tune Max (CTMax) */
        CtMax = PDE_XCVR_GET_CTUNE_SELECTED();

        /* Warm down the radio */
        PDE_AutoTxWarmDown();

        /* Calculate CTSlope */
        FreqDifference = (float)(EndFrequency - StartFreq);
        CtSlope = (float)(CtMax - CtMin) / FreqDifference;
        /* Calculate CT0 */
        CtInit = (float)CtMin + (float)BestDiffMin * CtSlope;

        while(FreqOffset < NumberOfFrequencies)
        {
            NewFreqCalc = StartFreq + (FreqStep * FreqOffset);

            CoarseTune = CtInit + (float)(NewFreqCalc - StartFreq) * CtSlope;

            CoarseLut[FreqOffset] = (uint8_t)lround(CoarseTune);

            FreqOffset++;
        }

        /* restore the previous frequency */
        (void)GENFSK_SetChannelNumber(GfskInstance,CurrentChannel);
    }
}
#endif /* PDE_USE_ADVANCED_CORSE_TUNE_COMPUTATION */

static void PDE_FrequencySweepInit (void){
    tpm_config_t fsweepTpm = {
        .prescale = kTPM_Prescale_Divide_1,
        .useGlobalTimeBase = false,
        .triggerSource = kTPM_TriggerSource_Internal,   //Use input capture channels
        .triggerSelect = kTPM_Trigger_Select_1,         //TPM_CH0 Input Capture
        .enableDoze = false,
        .enableDebugMode = true,
        .enableReloadOnTrigger = false,
        .enableStopOnOverflow = false,
        .enableStartOnTrigger = true,
    };

    /* Backup TSM Registers */
    PDE_TsmOverrideBackup();

    //Get the look-up table of coarse frequency configurations
#ifdef PDE_USE_ADVANCED_CORSE_TUNE_COMPUTATION
    PDE_CalculateCoarseTuneEnhanced(gPdeGfskInstanceId, &gPdeCoarseTuneConfigTable[0], gPdeSysCfg.freqSweepStartHz,
                                    gPdeSysCfg.freqSweepIndentHz, gPdeSysCfg.freqSweepNb);
#else
    /* @TODO: Use this method until the other one is debugged */
    PDE_CalculateCoarseTune(gPdeGfskInstanceId, &gPdeCoarseTuneConfigTable[0], gPdeSysCfg.freqSweepStartHz,
                            gPdeSysCfg.freqSweepIndentHz, gPdeSysCfg.freqSweepNb);
#endif

    //Get the look-up table of fine frequency configurations with carrier offset
    for(uint8_t iteration=0; iteration<gPdeSysCfg.freqSweepNb; iteration++){
        PDE_CalculateFineTune((gPdeSysCfg.freqSweepStartHz+(iteration*gPdeSysCfg.freqSweepIndentHz))+PDE_FREQ_SWEEP_OFFSET_HZ,\
                               g_xtal0Freq, &gPdeFineTuneConfigTable[1][iteration]);
    }

    //Get the look-up table of fine frequency configurations without carrier offset
    for(uint8_t iteration=0; iteration<gPdeSysCfg.freqSweepNb; iteration++){
        PDE_CalculateFineTune((gPdeSysCfg.freqSweepStartHz+(iteration*gPdeSysCfg.freqSweepIndentHz)), g_xtal0Freq,\
                               &gPdeFineTuneConfigTable[0][iteration]);
    }

    //Initialize timer

    /* Set IRQ Handler */
    OSA_InstallIntHandler((uint32_t)PDE_TIMER_IRQ, PDE_TPM_IRQHandler);

#if defined(RADIO_IS_GEN_3P5)
    /* TPM2CH0SRC conected to tof_timestamp_trg
       TX_MODE: tx_dig_en (from TSM) or pa_wu_complete (from TXDIG
                TOF_TX_SEL
                    0b - TSM: tx_dig_en (default at reset)
                    1b - TXDIG: pa_wu_complete
       RX_MODE: TOF_RX_SEL Time-of-Flight RX Select
                    0b - PHY: aa_fnd_to_ll (default at reset)
                    1b - Localization Control: pattern_found
    */
    /* Setup source of RADIO_TOF_TS_TRIG  */

    /* tx_dig_enable is the trigger in Tx mode 
       aa_fnd_to_ll is the trigger in Rx mode */
    XCVR_MISC->XCVR_CTRL = (XCVR_MISC->XCVR_CTRL & ~(XCVR_MISC_XCVR_CTRL_TOF_TX_SEL_MASK | XCVR_MISC_XCVR_CTRL_TOF_RX_SEL_MASK)) |
                           XCVR_MISC_XCVR_CTRL_TOF_TX_SEL(0U) |
                           XCVR_MISC_XCVR_CTRL_TOF_RX_SEL(0U);

    /* Connect RADIO_TOF_TS_TRIG signal to TPM2_CH0 (set as source) */
    SIM->SOPT4 |= SIM_SOPT4_TPM2CH0SRC(0x02U);

    /* TODO: workaround for XCVR HW bug TKT0515716.
     * coex clock needs to be active for tof_ts_trig to be propagated to SoC.
     * Should be removed when HW fix available.
     */
    XCVR_MISC->COEX_CTRL |= XCVR_MISC_COEX_CTRL_RF_NOT_ALLOWED_EN(0x2U);

    /* Select XTAL as the TPM clock source */
    CLOCK_SetTpmClock(2U);
    
#endif
    
    TPM_Init(PDE_TIMER, &fsweepTpm);
    TPM_EnableInterrupts(PDE_TIMER, (uint32_t)kTPM_TimeOverflowInterruptEnable);
    (void)EnableIRQ(PDE_TIMER_IRQ);

#if !defined(RADIO_IS_GEN_3P5)
    //Initialize trigger for DTEST
    PORT_SetPinMux(PORTB, 18U, kPORT_MuxAlt5);   //TPM0_CH0 PTB18, J4-2
    TPM_SetupInputCapture(PDE_TIMER, kTPM_Chnl_0, kTPM_RisingEdge);

    //Initialize DTEST13 pin
    PORT_SetPinMux(PORTB, 2U, kPORT_MuxAlt6);

#else
    /* TPM2CH0SRC conected to tof_timestamp_trg
       TX_MODE: tx_dig_en (from TSM) or pa_wu_complete (from TXDIG
                TOF_TX_SEL
                    0b - TSM: tx_dig_en (default at reset)
                    1b - TXDIG: pa_wu_complete
       RX_MODE: TOF_RX_SEL Time-of-Flight RX Select
                    0b - PHY: aa_fnd_to_ll (default at reset)
                    1b - Localization Control: pattern_found
    */
    #if 1
   
        TPM_SetupInputCapture(PDE_TIMER, kTPM_Chnl_0, kTPM_RisingEdge);
        
        //Initialize DTEST11 pin: J2-10
        PORT_SetPinMux(PORTB, 0U, kPORT_MuxAlt6);
        
    #else
        //Initialize trigger for DTEST
        PORT_SetPinMux(PORTB, 18U, kPORT_MuxAlt5);   //TPM0_CH0 PTB18, J4-2
        TPM_SetupInputCapture(PDE_TIMER, kTPM_Chnl_0, kTPM_RisingEdge);

        //Initialize DTEST11 pin: J2-10
        PORT_SetPinMux(PORTB, 0U, kPORT_MuxAlt6);
    #endif
#endif
}

static void PDE_FrequencySweepStart (void){
  
#if defined(RADIO_IS_GEN_3P5)
    /* Connect RADIO_TOF_TS_TRIG signal to TPM2_CH0 (set as source) */
    SIM->SOPT4 |= SIM_SOPT4_TPM2CH0SRC(0x02U);
#endif
  
    if(gPdeCurrentRole == kPdeRole_Rx_c){
        //Set frequency switch time for f0
        gFirstFsweepCounts = (uint32_t)PDE_F0_SWITCH_COUNT_RX;

        /* Configure DTEST signal that will start the frequency sweep timer */
        PDE_ENABLE_RX_TRIGGER;
    }
    else{
        //Set frequency switch time for f0
        gFirstFsweepCounts = (uint32_t)PDE_F0_SWITCH_COUNT_TX;

        /* Configure DTEST signal that will start the frequency sweep timer */
        PDE_ENABLE_TX_TRIGGER;
    }

    //Set frequency switch counts for fn */
    gNextFsweepCounts = PDE_FN_SWITCH_COUNT;

    //Set TPM period to first frequency sweep
    TPM_SetTimerPeriod(PDE_TIMER, gFirstFsweepCounts);
    PDE_TIMER->CNT = 0;

    //Start timer
    TPM_StartTimer(PDE_TIMER, kTPM_SystemClock);
}

static void PDE_FrequencySweepStop (void){

    //Stop Timer
    TPM_StopTimer(PDE_TIMER);

    //Release PLL Manual
    PDE_PllManualControlRelease();


    /* Disable DTEST pin */
    PDE_DISABLE_TRIGGER;
}

static inline void PDE_GetManualCalibrationValues(void)
{
    /* Obtain manual DC calibration values */
    PDE_XcvrGetManualCalibrationValues(&gPdeCalibration[gPdeCalibrationIndex++]);

    if(gPdeCalibrationIndex>=PDE_CALIBRATION_POINTS){
        gPdeCalibrationDone = true;
    }
}

static inline void PDE_ApplyManualCalibration(void)
{
    uint8_t counter, pickedIndex;

    /* Pick the smallest AGC index */
    pickedIndex = gPdeCalibration[0].agcIndex;
    gPdeCalibrationIndex = 0;
    for(counter=1; counter<PDE_CALIBRATION_POINTS; counter++){
        if(gPdeCalibration[counter].agcIndex<pickedIndex){
            pickedIndex = gPdeCalibration[counter].agcIndex;
            gPdeCalibrationIndex = counter;
        }
    }

    /* Apply the manual calibration values for the selected index */
    PDE_XcvrApplyManualCalibrationValues(&gPdeCalibration[gPdeCalibrationIndex]);
}

static inline void PDE_ClearManualCalibration (void){

    PDE_XcvrRestoreCalibrationSettings(&gPdeCalibration[gPdeCalibrationIndex]);

    /* Reset AGC */
    XCVR_ClearManAgc();
}

static void PDE_StartGenfskTx (pde_genfsk_tx_packet_type_t packetType, uint8_t *pData, uint16_t elementCount, uint64_t startDelayUs){
    static uint16_t pTempBuffer[PDE_MAX_ELEMENTS_PER_GFSK_TRANSACTION+5U];
    static uint16_t tempBufferSize, retryCount;
    uint16_t ackBuffer[2];
    genfskStatus_t status;
    GENFSK_packet_t pdeStartPacket = {
        .header.h0Field = 0,
        .header.h1Field = 0,
        .addr = gPdeSysCfg.peerAccessAddress,
    };

    /* Parameter not used, to be removed */
    assert(startDelayUs == 0UL);
    NOT_USED(startDelayUs);

    /* Abort any pending transaction */
    GENFSK_AbortAll();

    /* Configure the transmitter in the PDE_GENFSK_COMMUNICATIONS_CHANNEL */
    (void)GENFSK_SetChannelNumber(gPdeGfskInstanceId, PDE_GENFSK_COMMUNICATIONS_CHANNEL);

    if(packetType == kPdeGenfskTxPacketType_FirstTxAttempt_c){
        /* This is the first time this packet is tried to be sent */
        assert(elementCount<=PDE_MAX_ELEMENTS_PER_GFSK_TRANSACTION);

        /* Reset the retry count */
        retryCount = 0;

        //Configure packet header
        pdeStartPacket.header.lengthField = (elementCount*sizeof(uint16_t));
        pdeStartPacket.payload = pData;

        //Calculate the total byte count of the data to send
        tempBufferSize = (uint16_t)(pdeStartPacket.header.lengthField + sizeof(pdeStartPacket.addr) + 1U);

        /* Prepare message */
        status =  GENFSK_PacketToByteArray(gPdeGfskInstanceId, &pdeStartPacket, (uint8_t*)&pTempBuffer[0]);
        assert(status == gGenfskSuccess_c);
        NOT_USED(status);

        (void)PDE_SetNewAction(kPdeAction_WaitForAck_c);

        status = GENFSK_StartTx(gPdeGfskInstanceId, (uint8_t*)&pTempBuffer[0], tempBufferSize, 0UL);
        assert(status == gGenfskSuccess_c);
        NOT_USED(status);
    }
    else if (packetType == kPdeGenfskTxPacketType_Retry_c){
        /* This is a retry */
        retryCount++;   //Increase the retry count

        if(retryCount<PDE_MAX_RETRY_COUNT){
            /* Re-send the packet */
            (void)PDE_SetNewAction(kPdeAction_WaitForAck_c);

            status = GENFSK_StartTx(gPdeGfskInstanceId, (uint8_t*)&pTempBuffer[0], tempBufferSize, 0UL);
            assert(status == gGenfskSuccess_c);
        }
        else{
            gPdeReportedError = kPdeError_MaxRetriesReached_c;
            (void)OSA_EventSet(gPdeEventId, PDE_EVT_INTERNAL_ERROR);
        }
    }
    else{
        /* This is an acknowledge packet */
        ackBuffer[0] = (uint16_t)kPdeMessagingOpCode_Ack_c;
        ackBuffer[1] = (uint16_t)pData[0] + ((uint16_t)pData[1]<<8U);

        //Configure packet header
        pdeStartPacket.header.lengthField = 4; /* ACK OpCode + Responded OpCode */
        pdeStartPacket.payload = (uint8_t*)&ackBuffer[0];

        //Calculate the total byte count of the data to send
        tempBufferSize = (uint16_t)(pdeStartPacket.header.lengthField + sizeof(pdeStartPacket.addr) + 1U);

        /* Prepare message */
        status =  GENFSK_PacketToByteArray(gPdeGfskInstanceId, &pdeStartPacket, (uint8_t*)&pTempBuffer[0]);
        assert(status == gGenfskSuccess_c);

        /* Send message */
        status = GENFSK_StartTx(gPdeGfskInstanceId, (uint8_t*)&pTempBuffer[0], tempBufferSize, 0UL);
        assert(status == gGenfskSuccess_c);
    }
}

static void PDE_StartGenfskRx (uint32_t timeoutUs){
    genfskStatus_t status;
    /* Configure the device for waiting for a command */

    /* Cancel any pending GENFSK transaction */
    if( (GENFSK->XCVR_CTRL & GENFSK_XCVR_CTRL_XCVR_BUSY_MASK) != 0U ){
        GENFSK_AbortAll();
    }

    /* Configure the receiver in the PDE_GENFSK_COMMUNICATIONS_CHANNEL */
    status = GENFSK_SetChannelNumber(gPdeGfskInstanceId, PDE_GENFSK_COMMUNICATIONS_CHANNEL);
    assert(status == gGenfskSuccess_c);
    NOT_USED(status);

    /* Start RX */
    status = GENFSK_StartRx(gPdeGfskInstanceId, (uint8_t*)pPdeRxBuffer16, PDE_RX_BUFFER_SIZE, 0, timeoutUs);
    assert(status == gGenfskSuccess_c);
    NOT_USED(status);
}

static void PDE_SendStartRequest (void){
    uint16_t syncPayload[2] = {(uint16_t)kPdeMessagingOpCode_StartTest_c, gPdeSysCfg.nbOfSamplesPerFrequency};

    gPdeMeasurementSequenceStage = kPdeMeasurementSequence_Start_c;

    /* Reset the local sequence number */
    gPdeLocalSequenceNumber = 0;

#if PDE_DEBUG_START_CAL_TIMING
    D0_SET();
#endif

    (void)PDE_SetNewAction(kPdeAction_WaitForAck_c);

    /* Send start test request */
    PDE_StartGenfskTx(kPdeGenfskTxPacketType_FirstTxAttempt_c, (uint8_t*)&syncPayload, 2, 0);
}

static void PDE_DataRequest(pde_data_type_t dataType){
    uint16_t tempTxBuffer[3];

    /* Fill the TX buffer */
    tempTxBuffer[0] = (uint16_t)kPdeMessagingOpCode_DataRequest_c;
    tempTxBuffer[1] = gPdeLocalSequenceNumber;
    tempTxBuffer[2] = (uint16_t)dataType;

    /* Send packet */
    PDE_StartGenfskTx(kPdeGenfskTxPacketType_FirstTxAttempt_c, (uint8_t*)&tempTxBuffer[0], sizeof(tempTxBuffer)/sizeof(uint16_t), 0);
    gPdeLocalSequenceNumber++;
}

static void PDE_DataResponse (pde_data_type_t dataType, int16_t* pData, uint16_t offset, uint16_t dataCount){
    /* @Warning: For memory efficiency, this function assumes that the buffer holding pData will not be
    destroyed until all the data has been transmitted */
    uint16_t elementCount;
    uint16_t remainingData;
    uint16_t tempMessage[PDE_MAX_ELEMENTS_PER_GFSK_TRANSACTION];
    int16_t *pDataWithOffset = (int16_t*)(pData + offset);

    elementCount = (dataCount>PDE_MAX_ELEMENTS_PER_GFSK_TRANSACTION-5U) ? (PDE_MAX_ELEMENTS_PER_GFSK_TRANSACTION-5U) : (dataCount);
    remainingData = (dataCount>PDE_MAX_ELEMENTS_PER_GFSK_TRANSACTION-5U) ? (dataCount-elementCount) : (0U);

    /* Fill message buffer */
    tempMessage[0] = (uint16_t)kPdeMessagingOpCode_DataResponse_c;
    tempMessage[1] = gPdeLocalSequenceNumber;
    tempMessage[2] = (uint16_t)dataType;
    tempMessage[3] = elementCount;
    tempMessage[4] = remainingData;
    for(uint16_t index=0; index<PDE_MAX_ELEMENTS_PER_GFSK_TRANSACTION-5U; index++){
        tempMessage[(index+5U)] = (uint16_t)pDataWithOffset[index];
    }

    /* Send Packet */
    PDE_StartGenfskTx(kPdeGenfskTxPacketType_FirstTxAttempt_c, (uint8_t*)tempMessage, (elementCount+5U), 0);
    gPdeLocalSequenceNumber++;

    /* Send a self message indicating the remaining data to send */
    if(remainingData != 0U){
        gPdeRemainingData.isDataPending = true;
        gPdeRemainingData.dataType = dataType;
        gPdeRemainingData.remainingElements = remainingData;
        gPdeRemainingData.dataPointer = pData;
        if(offset == 0U){
            gPdeRemainingData.originalElementCount = dataCount; //Store the original number of elements only
        }
    }
    else{
        gPdeRemainingData.isDataPending = false;
    }
}

static void PDE_GfskAccessAddressMatchEvent (pde_action_t actionToExecute){
#if PDE_DEBUG_START_CAL_TIMING
    D1_CLEAR();
#endif
    /* Capture the calibration values*/
    PDE_GetManualCalibrationValues();
}

static void PDE_GfskPacketTransmittedEvent (pde_action_t actionToExecute){
    if(actionToExecute == kPdeAction_StartGenfskRx_c){
        PDE_StartGenfskRx(PDE_TIMEOUT_DEFAULT_US);
    }

    if(actionToExecute == kPdeAction_WaitForAck_c){
        PDE_StartGenfskRx(PDE_TIMEOUT_ACK_US);
    }

    if(actionToExecute == kPdeAction_StartPdeSyncRx_c){
        PDE_StartSyncRx();
    }

    if(actionToExecute == kPdeAction_StartCalibrationPacketTx_c){
        /* Start Calibration Packet Transmission */
        (void)PDE_SetNewAction(kPdeAction_WaitForAck_c);  //Indicate we are expecting an ACK for this packet
        PDE_TimeDelayUs(150);   //Wait 150uS for the peer device to be ready
        PDE_StartCalibration(kPdeStartCalibration_Tx_c, gPdeCalibrationChannel[gPdeCalibrationChannelIndex]); //Send the calibration packet
        if(gPdeCurrentDeviceType == kPdeDeviceType_RD_c){
            /* Update the calibration channel */
            gPdeCalibrationChannelIndex++;
        }
    }

    if(actionToExecute == kPdeAction_StartCalibrationPacketRx_c){
        /* Start Calibration Packet Reception */
        PDE_StartCalibration(kPdeStartCalibration_Rx_c, gPdeCalibrationChannel[gPdeCalibrationChannelIndex]);
    }
    if(actionToExecute == kPdeAction_ParseDataRequestOnAckSent_c){
        /* Allow peer device to start RX */
        PDE_TimeDelayUs(50);

        PDE_GenfskPacketParseRequest(&pPdeRxBuffer16[2]);
    }

    if(actionToExecute == kPdeAction_ParseDataResponseOnAckSent_c){
        PDE_GenfskPacketParseResponse(&pPdeRxBuffer16[2]);
    }

}

/* process ack packet */
static void PDE_GfskPacketReceivedEvent_Ack (void){
  
    pde_messaging_opcode_t ackedOpCode = (pde_messaging_opcode_t)pPdeRxBuffer16[1];

    switch(ackedOpCode){
      case kPdeMessagingOpCode_StartTest_c:{
          /* Request ACK on Calibration Packet TX completion */
          (void)PDE_SetNewAction(kPdeAction_WaitForAck_c);
          /* Send calibration TX packet */
          PDE_StartCalibration(kPdeStartCalibration_Tx_c, gPdeCalibrationChannel[gPdeCalibrationChannelIndex]);
      }
      break;

      case kPdeMessagingOpCode_StartCalibration_c:{
          /* Calibration packet was received by the peer */

          if(gPdeCurrentDeviceType == kPdeDeviceType_MD_c){
              /* MD Sends the packet first. Start the calibration packet reception */
              PDE_StartCalibration(kPdeStartCalibration_Rx_c, gPdeCalibrationChannel[gPdeCalibrationChannelIndex]);
              /* Update calibration channel to the next one */
              gPdeCalibrationChannelIndex++;
          }
          else{
              if(gPdeCalibrationDone){
                  /* Apply the Calibration */
                  PDE_ApplyManualCalibration();
                  /* RD, send the Sync TX */
                  PDE_TimeDelayUs(100);    //Wait 100uS for peer to be ready
                  PDE_StartSyncTx();
              }
              else{
                  /* Start a new calibration RX sequence */
                  PDE_StartCalibration(kPdeStartCalibration_Rx_c, gPdeCalibrationChannel[gPdeCalibrationChannelIndex]);
              }
          }
      }
      break;

      case kPdeMessagingOpCode_DataRequest_c:{
          /* Wait for data */
          PDE_StartGenfskRx(PDE_TIMEOUT_DEFAULT_US);
      }
      break;

      case kPdeMessagingOpCode_DataResponse_c:{
          /* Check if there is more data to send */
          if(gPdeRemainingData.isDataPending){
              /* Calculate offset and send remaining data */
              uint16_t dataOffset = gPdeRemainingData.originalElementCount-gPdeRemainingData.remainingElements;
              PDE_DataResponse(gPdeRemainingData.dataType, gPdeRemainingData.dataPointer, dataOffset, gPdeRemainingData.remainingElements);
          }
          else{
              if(gPdeRemainingData.dataPointer == &gPdeLocalPhaseBuffer[0]){
                  /* Clear Phase Buffer */
                  for(uint32_t index=0; index<(sizeof(gPdeLocalPhaseBuffer)/sizeof(uint16_t)); index++){
                      gPdeLocalPhaseBuffer[index]=0;
                  }
              }
              else{
                  /* Clear I or Q buffer */
               }
              /* Return to wait mode */
              PDE_StartGenfskRx(PDE_TIMEOUT_WAIT_FOREVER);
          }
      }
      break;

      /* LCOV_EXCL_START */
      default:{
          /* Wrong code. Report error */
          gPdeReportedError = kPdeError_UnknownOpCode_c;
          (void)OSA_EventSet(gPdeEventId, PDE_EVT_INTERNAL_ERROR);
      }
      break;
      /* LCOV_EXCL_STOP */
    }  
}

static void PDE_GfskPacketReceivedEvent (pde_action_t actionToExecute){
    /* Parse Received Message */

    /* Retrieve OpCode */
    pde_messaging_opcode_t opCode = (pde_messaging_opcode_t)pPdeRxBuffer16[0];

    switch(opCode){
      case kPdeMessagingOpCode_Ack_c: {
          PDE_GfskPacketReceivedEvent_Ack();
      }
      break;

      case kPdeMessagingOpCode_StartTest_c:{
          pde_messaging_opcode_t opCode2 = kPdeMessagingOpCode_StartTest_c;
          
          /* Update the number of samples per frequency */
          gPdeSysCfg.nbOfSamplesPerFrequency = (uint8_t)pPdeRxBuffer16[1];

          /* Reset the local sequence number */
          gPdeLocalSequenceNumber = 0;

          /* Reset the Calibration Index */
          gPdeCalibrationIndex = 0;
          gPdeCalibrationChannelIndex = 0;
          gPdeCalibrationDone = false;

          /* Indicate Start with LED */
          GPIOC->PSOR |= (1U<<1U);

          /* Configure the current device type */
          gPdeCurrentDeviceType = kPdeDeviceType_RD_c;
          gPdeCurrentRole = kPdeRole_Tx_c;

          /* Obtain the number of samples to capture */
          gPdeRemainingDmaCaptures = gPdeSysCfg.freqSweepNb;
          gRequestedSamples = PDE_MAX_NUMBER_OF_TONE_IQ_SAMPLES + 
							  gPdeSysCfg.nbOfSamplesPerFrequency * (gPdeRemainingDmaCaptures-1U);
 
          /* Start Timeout Timer */
          PDE_StartTimeoutTimer(PDE_TIMEOUT_GLOBAL_MS);

          /* Indicate to start calibration RX after ACK transmission */
          (void)PDE_SetNewAction(kPdeAction_StartCalibrationPacketRx_c);
          PDE_StartGenfskTx(kPdeGenfskTxPacketType_Ack_c, (uint8_t*)&opCode2, 1, 0);
      }
      break;

      case kPdeMessagingOpCode_StartCalibration_c:{
          if(gPdeCalibrationDone){
              /* Calibration process is complete */
              if(gPdeCurrentDeviceType == kPdeDeviceType_MD_c){
                  /* Apply the Calibration */
                  PDE_ApplyManualCalibration();
                  /* Request receiving the sync packet after ACK has been sent */
                  (void)PDE_SetNewAction(kPdeAction_StartPdeSyncRx_c);
              }
              else{
                  /* Request sending the calibration packet on ACK sent */
                  (void)PDE_SetNewAction(kPdeAction_StartCalibrationPacketTx_c);
              }
          }
          else{
              /* Request sending the calibration packet on ACK sent */
              (void)PDE_SetNewAction(kPdeAction_StartCalibrationPacketTx_c);
          }
          /* Send ACK */
          PDE_StartGenfskTx(kPdeGenfskTxPacketType_Ack_c, (uint8_t*)&opCode, 1, 0);
      }
      break;

      case kPdeMessagingOpCode_DataRequest_c:{
          /* Data request received. Store data */
          uint16_t remoteSequenceNumber = pPdeRxBuffer16[1];
          uint16_t requestedDataType = pPdeRxBuffer16[2];

          if(remoteSequenceNumber > gPdeLocalSequenceNumber){
              /* Report mismatch error */
              gPdeReportedError = kPdeError_SnMismatch_c;
              (void)OSA_EventSet(gPdeEventId, PDE_EVT_INTERNAL_ERROR);
          }
          else if (remoteSequenceNumber < gPdeLocalSequenceNumber){
              /* This is a retried packet. We got the previous but peer didn't got the ACK. Just send ACK again*/
              (void)PDE_SetNewAction(kPdeAction_StartGenfskRx_c); //Request to go RX again after sending ACK
              PDE_StartGenfskTx(kPdeGenfskTxPacketType_Ack_c, (uint8_t*)&opCode, 1, 0);
          }
          else{
              /* Check if the requested data type is valid */
              if(requestedDataType < (uint16_t)kPdeDataType_Count_c){
                  /* Valid request parse on ACK TX completion*/
                  (void)PDE_SetNewAction(kPdeAction_ParseDataRequestOnAckSent_c);

                  /* Send ACK */
                  PDE_StartGenfskTx(kPdeGenfskTxPacketType_Ack_c, (uint8_t*)&opCode, 1, 0);

                  /* Increase local SN */
                  gPdeLocalSequenceNumber++;
              }
          }
      }
      break;

      case kPdeMessagingOpCode_DataResponse_c:{
          /* Data request responded. Store data */
          uint16_t remoteSequenceNumber = pPdeRxBuffer16[1];
          uint16_t requestedDataType = pPdeRxBuffer16[2];

          if(remoteSequenceNumber > gPdeLocalSequenceNumber){
              /* Report mismatch error */
              gPdeReportedError = kPdeError_SnMismatch_c;
              (void)OSA_EventSet(gPdeEventId, PDE_EVT_INTERNAL_ERROR);
          }
          else if (remoteSequenceNumber < gPdeLocalSequenceNumber){
               /* This is a retried packet. We got the previous but peer didn't got the ACK. Just send ACK again*/
              (void)PDE_SetNewAction(kPdeAction_StartGenfskRx_c); //Request to go RX again after sending ACK
              PDE_StartGenfskTx(kPdeGenfskTxPacketType_Ack_c, (uint8_t*)&opCode, 1, 0);
          }
          else{
              /* Check if the requested data type is valid */
              if(requestedDataType < (uint16_t)kPdeDataType_Count_c){
                  /* Valid request parse on ACK TX completion*/
                  (void)PDE_SetNewAction(kPdeAction_ParseDataResponseOnAckSent_c);

                  /* Send ACK */
                  PDE_StartGenfskTx(kPdeGenfskTxPacketType_Ack_c, (uint8_t*)&opCode, 1, 0);

                  /* Increase local SN */
                  gPdeLocalSequenceNumber++;
              }
          }
      }
      break;

      default:{
          /* Wrong code. Report error */
          gPdeReportedError = kPdeError_UnknownOpCode_c;
          (void)OSA_EventSet(gPdeEventId, PDE_EVT_INTERNAL_ERROR);
      }
      break;
    }
}

/*
 * Sequence:
 *   kPdeAction_ReportPhaseDataReady_c
 *   kPdeAction_ReportFreqMaskReady_c
 *   Optionaly (PDE_REPORT_RAW_IQ)
 *     [kPdeAction_ReportIDataReady_c
 *     kPdeAction_ReportQDataReady_c
 *   kPdeEvent_CaptureComplete_c | kPdeEvent_ErrorOccurred_c
 */
static void PDE_DataRequestCompleteEvent (pde_action_t actionToExecute){
    if(actionToExecute == kPdeAction_ReportIQDataReady_c){
        /* Notify  data is ready */
        gPdeSysCfg.pdeEventCallback(kPdeEvent_CaptureComplete_c, kPdeError_Ok_c);

        /* Restart globals */
        PDE_ResetSystemToDefault();

        if (gPdeCurrentDeviceType == kPdeDeviceType_RD_c) {
            /* Start wait mode */
            PDE_StartGenfskRx(PDE_TIMEOUT_WAIT_FOREVER);
        }
    }

    if(actionToExecute == kPdeAction_ReportPhaseDataReady_c){
        assert(pPdeCombinedPhaseBuffer);

        if ((gPdeCaptureCfg.reportFlags & PDE_REPORT_RAW_IQ) != 0U) {
            /* Backup preprocessed phases before doing the combination algorithm */
            PDEMemCpy(&aPdeLastRdPreprocessedPhase[0], &gPdeRemotePhaseBuffer[0], sizeof(aPdeLastRdPreprocessedPhase));
            PDEMemCpy(&aPdeLastMdPreprocessedPhase[0], &gPdeLocalPhaseBuffer[0], sizeof(aPdeLastMdPreprocessedPhase));
        }

        /* Request Frequency Mask */

        /* Wait 100uS for peer to start RX */
        PDE_TimeDelayUs(100);

        PDE_DataRequest(kPdeDataType_FreqMaskData_c);
    }

    if(actionToExecute == kPdeAction_ReportFreqMaskReady_c){
        bool isMeasurementValid;

        /* Call the Phase combination algorithm */
        PDE_PreprocessingCombinePhases(&gPdeRemotePhaseBuffer[0], &gPdeLocalPhaseBuffer[0], pPdeCombinedPhaseBuffer,
                                       (gPdeSysCfg.freqSweepNb - 1U));

        /* Free Phase samples buffer */
        for(uint16_t index=0; index<(sizeof(gPdeRemotePhaseBuffer)/sizeof(uint16_t)); index++){
            gPdeRemotePhaseBuffer[index]=0;
            gPdeLocalPhaseBuffer[index]=0;
        }

        /* Call the Freq Mask combination algorithm */
        isMeasurementValid = PDE_PreprocessingCombineFreqMasks(&FreqMaskLocal[0],&FreqMaskPeer[0],pPdeFreqMask,
                                                               (gPdeSysCfg.freqSweepNb - 1U));

        /* Just process if combined freq mask allows us to continue, report error otherwise */
        if(isMeasurementValid == true)
        {
            if ((gPdeCaptureCfg.reportFlags & PDE_REPORT_RAW_IQ) != 0U)
            {
                /* Request IQ data from RD. DEMO LOG mode can skip the masurement validation since this data is used for debug */
                PDE_DataRequest(kPdeDataType_IQData_c);
            }
            else
            {
                /* Report acquisition complete */
                gPdeSysCfg.pdeEventCallback(kPdeEvent_CaptureComplete_c, kPdeError_Ok_c);
                
                /* Restart globals */
                PDE_ResetSystemToDefault();
            }
        }
        else
        {
            gPdeSysCfg.pdeEventCallback(kPdeEvent_ErrorOccurred_c, kPdeError_PhaseInvalid_c);
        }

        pPdeCombinedPhaseBuffer = NULL;
    }
}

static void PDE_GenfskPacketParseRequest (uint16_t *pInPacket){
    pde_data_type_t packetType = (pde_data_type_t)pInPacket[0];

    /* Allow some uS so peer device can prepare RX the first time */
    PDE_TimeDelayUs(50);

    if(packetType == kPdeDataType_IQData_c){
        if(gSamplesInBuffer != 0U){
            /* Send IQ samples over the air */
            PDE_DataResponse(kPdeDataType_IQData_c, &gIqSamplesBuffer[0], 0, gSamplesInBuffer*2U);
        }
    }

    if(packetType == kPdeDataType_PhaseData_c){
        /* Send Phase samples over the air */
        PDE_DataResponse(kPdeDataType_PhaseData_c, &gPdeLocalPhaseBuffer[0], 0, gPdeSysCfg.freqSweepNb-1U);
    }

    if(packetType == kPdeDataType_FreqMaskData_c){
        /* Send Frequency Mask Over the Air */
        PDE_DataResponse(kPdeDataType_FreqMaskData_c, (int16_t*)(void*)&FreqMaskLocal[0], 0, (sizeof(FreqMaskLocal)/sizeof(uint16_t)));
    }
}

/*
 * On MD device:
 * Process response to DataRequest
 * Generate event PDE_EVT_DATA_REQUEST_COMPLETE, with newAction set to one of:
 * - kPdeAction_ReportIQDataReady_c
 * - kPdeAction_ReportPhaseDataReady_c
 * - kPdeAction_ReportFreqMaskReady_c
 */
static void PDE_GenfskPacketParseResponse (uint16_t *pInPacket){
    /* Get Data Type */
    pde_data_type_t dataType = (pde_data_type_t)pInPacket[0];

    if(dataType == kPdeDataType_IQData_c){
        static uint16_t dataOffset = 0;

        uint16_t receivedData = pInPacket[1];
        uint16_t remainingData = pInPacket[2];

        /* Extract Data */
        /* Ensure we do not exceed IQ buffer.
         * It can happen in case a paquet is retried.
         * TODO: We need a sequence counter in the paquet header to know where to store the data.
         */
        assert(dataOffset + receivedData <= (PDE_MAX_IQ_SAMPLES*2U));
        for(uint16_t index = 0; index<receivedData; index++){
            gPdePeerIQDataBuffer[index+dataOffset] = (int16_t)pInPacket[index + 3U];
        }

        dataOffset += receivedData;

        if(remainingData == 0U){
            /* Notify the data is ready */
            (void)PDE_SetNewAction(kPdeAction_ReportIQDataReady_c);

            (void)OSA_EventSet(gPdeEventId, PDE_EVT_DATA_REQUEST_COMPLETE);
            dataOffset = 0;
        }
        else{
            /* Start RX again to wait for remaining data */
            PDE_StartGenfskRx(PDE_TIMEOUT_DEFAULT_US);
        }
    }

    if(dataType == kPdeDataType_PhaseData_c){
        static uint16_t dataOffset = 0;

        uint16_t receivedData = pInPacket[1];
        uint16_t remainingData = pInPacket[2];

        /* Extract Data */
        for(uint16_t index = 0; index<receivedData; index++){
            gPdeRemotePhaseBuffer[index+dataOffset] = (int16_t)pInPacket[index + 3U];
        }

        dataOffset += receivedData;

        if(remainingData == 0U){
            /* Notify the data is ready */
            (void)PDE_SetNewAction(kPdeAction_ReportPhaseDataReady_c);

            (void)OSA_EventSet(gPdeEventId, PDE_EVT_DATA_REQUEST_COMPLETE);
            dataOffset = 0;
        }
        else{
            /* Start RX again to wait for remaining data */
            PDE_StartGenfskRx(PDE_TIMEOUT_DEFAULT_US);
        }
    }

    if(dataType == kPdeDataType_FreqMaskData_c){
        static uint16_t dataOffset = 0;
        uint16_t* pTempDataPointer = (uint16_t*)(void*)&FreqMaskPeer[0];

        uint16_t receivedData = pInPacket[1];
        uint16_t remainingData = pInPacket[2];

        /* Extract Data */
        for(uint16_t index = 0; index<receivedData; index++){
            pTempDataPointer[index+dataOffset] = pInPacket[index + 3U];
        }

        dataOffset += receivedData;

        if(remainingData == 0U){
            /* Notify the data is ready */
            (void)PDE_SetNewAction(kPdeAction_ReportFreqMaskReady_c);

            (void)OSA_EventSet(gPdeEventId, PDE_EVT_DATA_REQUEST_COMPLETE);
            dataOffset = 0;
        }
        else{
            /* Start RX again to wait for remaining data */
            PDE_StartGenfskRx(PDE_TIMEOUT_DEFAULT_US);
        }
    }
}

static uint8_t PDE_SetNewAction (pde_action_t newAction){
    uint8_t retVal = 1U;

    if(gPdeActionOnEvent == kPdeAction_NoAction_c){
        if(newAction <  kPdeAction_Critical_c || gPdeLastActionExecuted > kPdeAction_Critical_c){
            /* set the next action */
            gPdeActionOnEvent = newAction;
            
            /* OK */
            retVal = 0U;
        }
    }

    return retVal;
}

static void PDE_StartCalibration (pde_start_calibration_t calibrationPacketAction, uint8_t calibrationChannel){
    genfskStatus_t status;

    gPdeMeasurementSequenceStage = kPdeMeasurementSequence_Calibration_c;

    /* Abort any ongoing GFSK transaction */
    GENFSK_AbortAll();

    /* Configure Calibration Channel */
    (void)GENFSK_SetChannelNumber(gPdeGfskInstanceId, calibrationChannel);

    /* This device will send the sync packet */
    if(calibrationPacketAction == kPdeStartCalibration_Tx_c){
#if PDE_DEBUG_START_CAL_TIMING
        if(gPdeCurrentDeviceType == kPdeDeviceType_MD_c){
            D0_CLEAR();
        }
        else{
            D0_SET();
        }
#endif

        /* Request ACK after calibration packet is sent */
        (void)PDE_SetNewAction(kPdeAction_WaitForAck_c);

        /* Modify the OpCode bytes with the Calibration OpCode */
        aPdeCalibrationSyncPacket[5] = (uint8_t)((uint16_t)kPdeMessagingOpCode_StartCalibration_c&0x00FFU);
        aPdeCalibrationSyncPacket[6] = (uint8_t)(((uint16_t)kPdeMessagingOpCode_StartCalibration_c&0xFF00U)>>8U);

        status = GENFSK_StartTx(gPdeGfskInstanceId, &aPdeCalibrationSyncPacket[0], (uint16_t)sizeof(aPdeCalibrationSyncPacket), 0);
        assert(status == gGenfskSuccess_c);
        NOT_USED(status);
    }
    /*This device will receive the sync packet */
    else{
        uint32_t genfskEvents;

        /* Enable Access Address Match event temporally */
        genfskEvents = GENFSK_GetEventMask(gPdeGfskInstanceId);
        genfskEvents |= gGenfskNwkAddressMatch;
        (void)GENFSK_SetEventMask(gPdeGfskInstanceId, genfskEvents);

#if PDE_DEBUG_START_CAL_TIMING
        if(gPdeCurrentDeviceType == kPdeDeviceType_MD_c){
            D0_SET();
        }
        else{
            D0_CLEAR();
        }
#endif

        status = GENFSK_StartRx(gPdeGfskInstanceId, (uint8_t*)pPdeRxBuffer16, PDE_RX_BUFFER_SIZE, 0, PDE_TIMEOUT_DEFAULT_US);
        assert(status == gGenfskSuccess_c);
        NOT_USED(status);
#if (PDE_DEBUG_DC_RD || PDE_DEBUG_DC_MD)
        PDE_IqDmaStartCapture(32, &gIqSamplesBuffer32[0], kPdeDmaCapture_FirstRxExecution_c);
#endif
    }
}

static void PDE_StartSyncTx (void){
    genfskStatus_t status;

    gPdeMeasurementSequenceStage = kPdeMeasurementSequence_Sync_c;

    /* Abort any ongoing GFSK transaction */
    if((GENFSK->XCVR_CTRL & GENFSK_XCVR_CTRL_XCVR_BUSY_MASK) != 0U){
        GENFSK_AbortAll();
    }

    /* Prepare Frequency Sweep algorithm */
    PDE_FrequencySweepStart();

    /* Reset the DMA buffer */
    gSamplesInBuffer = 0;

    /* Configure start frequency */
#if !defined(RADIO_IS_GEN_3P5)
    (void)XCVR_OverrideChannel((uint8_t)((gPdeSysCfg.freqSweepStartHz/1000000U)-2360U), true);
#else
    (void)XCVR_OverrideChannel((uint8_t)((gPdeSysCfg.freqSweepStartHz/1000000U)-2360U));
#endif

    /* Disable CRC and Whiten as this packet will go CW TX/RX before the transmission is completed */
    pdeCrcConfig.crcEnable = gGenfskCrcDisable;
    status = GENFSK_SetCrcConfig(gPdeGfskInstanceId, &pdeCrcConfig);
    assert(status == gGenfskSuccess_c);
    pdeWhitenerConfig.whitenStart = gWhitenStartNoWhitening;
    status = GENFSK_SetWhitenerConfig(gPdeGfskInstanceId, &pdeWhitenerConfig);
    assert(status == gGenfskSuccess_c);

#if PDE_DEBUG_START_CAL_TIMING
    D0_CLEAR();
#endif

    /* Modify the OpCode bytes with the Sync OpCode */
    aPdeCalibrationSyncPacket[5] = (uint8_t)((uint16_t)kPdeMessagingOpCode_StartSync_c&0x00FFU);
    aPdeCalibrationSyncPacket[6] = (uint8_t)(((uint16_t)kPdeMessagingOpCode_StartSync_c&0xFF00U)<<8U);

    status = GENFSK_StartTx(gPdeGfskInstanceId, &aPdeCalibrationSyncPacket[0], (uint16_t)sizeof(aPdeCalibrationSyncPacket), 0);
    assert(status == gGenfskSuccess_c);
    NOT_USED(status);
}

static void PDE_StartSyncRx (void){
    genfskStatus_t status;
    
    gPdeMeasurementSequenceStage = kPdeMeasurementSequence_Sync_c;

    /* Configure start frequency */
#if !defined(RADIO_IS_GEN_3P5)
    (void)XCVR_OverrideChannel((uint8_t)(((gPdeSysCfg.freqSweepStartHz/1000000U)-2360U)),true);
#else
    (void)XCVR_OverrideChannel((uint8_t)((gPdeSysCfg.freqSweepStartHz/1000000U)-2360U));
#endif

    /* Before start, get the TSM configuration for TX */
    /* Start Auto TSM Warm Up Sequence */
    PDE_AutoTxWarmUp();
    /* Lock TX TSM */
    PDE_TsmOverrideTx();
    /* Warm down the radio */
    PDE_AutoTxWarmDown();

    if((GENFSK->XCVR_CTRL & GENFSK_XCVR_CTRL_XCVR_BUSY_MASK) != 0U){
        GENFSK_AbortAll();
    }

    /* Disable CRC and Whiten as this packet will go CW TX/RX before the transmission is completed */
    pdeCrcConfig.crcEnable = gGenfskCrcDisable;
    status = GENFSK_SetCrcConfig(gPdeGfskInstanceId, &pdeCrcConfig);
    assert(status == gGenfskSuccess_c);
    pdeWhitenerConfig.whitenStart = gWhitenStartNoWhitening;
    status = GENFSK_SetWhitenerConfig(gPdeGfskInstanceId, &pdeWhitenerConfig);
    assert(status == gGenfskSuccess_c);

#if PDE_DEBUG_START_CAL_TIMING
    D0_CLEAR();
#endif

    status = GENFSK_StartRx(gPdeGfskInstanceId, (uint8_t*)pPdeRxBuffer16, PDE_RX_BUFFER_SIZE, 0, PDE_TIMEOUT_WAIT_FOREVER);
    assert(status == gGenfskSuccess_c);
    NOT_USED(status);

#if (PDE_DEBUG_DC_MD)
    PDE_IqDmaStartCapture(PDE_MAX_NUMBER_OF_TONE_IQ_SAMPLES, &gIqSamplesBuffer32[0], kPdeDmaCapture_FirstRxExecution_c);
#else
    /* Reset the DMA buffer and start first capture*/
    gSamplesInBuffer = 0;
    PDE_IqDmaStartCapture(PDE_MAX_NUMBER_OF_TONE_IQ_SAMPLES, &gIqSamplesBuffer32[gSamplesInBuffer], kPdeDmaCapture_FirstRxExecution_c);
#endif

    /* Prepare frequency sweep algorithm */
    PDE_FrequencySweepStart();
}

static void PDE_InitTimeoutTimer(void){
    /* initialize the timer */
    TMR_Init();
    gPdeTimeoutTimerId = TMR_AllocateTimer();
    assert(gPdeTimeoutTimerId != gTmrInvalidId_c);
}

static void PDE_StartTimeoutTimer(uint32_t timeMs){
    /* start the timer */
    (void)TMR_StartSingleShotTimer(gPdeTimeoutTimerId, timeMs, PDE_TimeoutCallback, NULL);
}

static void PDE_StopTimeoutTimer(void){
    /* stop the timer */
    (void)TMR_StopTimer(gPdeTimeoutTimerId);
}

static void PDE_TimeoutCallback (void *pParam){
    /* handler timer expiry */
    (void)OSA_EventSet(gPdeEventId, PDE_EVT_INTERNAL_ERROR);
    gPdeReportedError = kPdeError_Timeout_c;
}

static void PDE_ResetRadioToDefaults(void){
    /* Reset radio to defauls */
    XCVR_Reset();

    /* Reset GFSK to defaults */
    GENFSK_ResetToDefaults(gPdeGfskInstanceId);

    /* Reinitialize radio */
#if !defined(RADIO_IS_GEN_3P5)
    XCVR_Init(GFSK_BT_0p5_h_0p5, DR_1MBPS);
#else
    const xcvr_config_t *p_xcvr_cfg = &xcvr_gfsk_bt_0p5_h_0p5_1mbps_full_config;
    const xcvr_coding_config_t *p_rbme_cfg = &xcvr_ble_coded_s2_config;
    (void)XCVR_Init(&p_xcvr_cfg, &p_rbme_cfg);
    (void)XCVR_SetActiveLL(XCVR_ACTIVE_LL_GENFSK);
#endif
    /* LCOV_EXCL_START */
    if(gHardwareParameters.xtalTrim != 0xFFFFFFFFU){
        /* Apply XTAL Trim only if it was stored previously */
        (void)XCVR_SetXtalTrim( (uint8_t)gHardwareParameters.xtalTrim );
    }
    /* LCOV_EXCL_STOP */
    (void)XCVR_ForcePAPower(0x3F);    //Max PA power during CW

    /* Reinitialize GFSK */
    (void)GENFSK_RadioConfig(gPdeGfskInstanceId, &pdeRadioConfig);
    (void)GENFSK_SetPacketConfig(gPdeGfskInstanceId, &pdePacketConfig);
#if PDE_ENABLE_CRC_FOR_GFSK
    pdeCrcConfig.crcEnable = gGenfskCrcEnable;
#else
    pdeCrcConfig.crcEnable = gGenfskCrcDisable;
#endif
    (void)GENFSK_SetCrcConfig(gPdeGfskInstanceId, &pdeCrcConfig);
    pdeWhitenerConfig.whitenStart = gWhitenStartWhiteningAtH0;
    (void)GENFSK_SetWhitenerConfig(gPdeGfskInstanceId, &pdeWhitenerConfig);
    (void)GENFSK_SetNetworkAddress(gPdeGfskInstanceId, 0, &pdeNetworkAddress);
    (void)GENFSK_EnableNetworkAddress(gPdeGfskInstanceId, 0);
    (void)GENFSK_SetTxPowerLevel(gPdeGfskInstanceId, 32); //Max output power
}

/*******************************************************************************
* Callback functions and ISR
*******************************************************************************/
static void PDE_RxCallback (uint8_t *pBuffer, uint16_t bufferLength, uint64_t timestamp, int8_t rssi, uint8_t crcValid){
    GENFSK_packet_t rxGenfskPacket;

    if(pdeCrcConfig.crcEnable == gGenfskCrcEnable && crcValid == 0U){
        /* Data is corrupted. Abort */
        (void)OSA_EventSet(gPdeEventId, PDE_EVT_INTERNAL_ERROR);
        gPdeReportedError = kPdeError_CrcInvalid_c;
    }
    else{
        /* Convert GENFSK packet to bytes */
        rxGenfskPacket.payload = (uint8_t*)&pPdeRxBuffer16[0];
        (void)GENFSK_ByteArrayToPacket(gPdeGfskInstanceId, pBuffer, &rxGenfskPacket);

        /* Notify PDE task */
        (void)OSA_EventSet(gPdeEventId, PDE_EVT_GFSK_PACKET_RECEIVED);
    }
}

static void PDE_EventCallback (genfskEvent_t event, genfskEventStatus_t eventStatus){
    /* GENFSK event callback */
    switch(event){
      case gGenfskTxEvent:{
          (void)OSA_EventSet(gPdeEventId, PDE_EVT_GFSK_PACKET_TRANSMITTED);
      }
      break;

      case gGenfskRxEvent:{
          /* Data is corrupted. Abort */
          (void)OSA_EventSet(gPdeEventId, PDE_EVT_INTERNAL_ERROR);
          if(eventStatus == gGenfskCRCInvalid){
              gPdeReportedError = kPdeError_CrcInvalid_c;
          }
          else if(eventStatus == gGenfskTimeout){
              gPdeReportedError = kPdeError_Timeout_c;
          }
          else{
              gPdeReportedError = kPdeError_GfsxRxError_c;
          }
      }
      break;

      case gGenfskNwkAddressMatch:{
          #if PDE_DEBUG_START_CAL_TIMING
          D1_SET();
          #endif
          /* Start an event to indicate the AA match */
          (void)OSA_EventSet(gPdeEventId, PDE_EVT_ACCESS_ADDRESS_MATCH);
          /* Disabe AA Match event */
          (void)GENFSK_SetEventMask(gPdeGfskInstanceId, gGenfskTxEvent | gGenfskRxEvent | gGenfskWakeEvent);
      }
      break;
      /* LCOV_EXCL_START */
      default: {
          assert(FALSE);
      }
      /* LCOV_EXCL_STOP */
      break;
    }
}

#if !defined(RADIO_IS_GEN_3P5)
static void PDE_DmaCallback (struct _edma_handle *handle, void *userData, bool transferDone, uint32_t tcds){
#else
static void PDE_DmaCallback (void *userData, bool transferDone, uint16_t count){
#endif

    /* Stop DMA in the XCVR */
    PDE_XcvrDmaStop();

    //Update the number of captured samples
    if( gSamplesInBuffer == 0U )
    {
        gSamplesInBuffer = PDE_MAX_NUMBER_OF_TONE_IQ_SAMPLES;
		(void)OSA_EventSet(gPdeEventId, PDE_EVT_CALIBRATE_DC);
    }
    else
    {
        gSamplesInBuffer += gPdeSysCfg.nbOfSamplesPerFrequency;
    }

    #if PDE_DEBUG_IQ_CAPTURE_TIMING
    D1_CLEAR();   //RX Capture complete instant
    #endif  //PDE_DEBUG_IQ_CAPTURE_TIMING


    if(!transferDone)
    {
        (void)OSA_EventSet(gPdeEventId, PDE_EVT_INTERNAL_ERROR);
        gPdeReportedError = kPdeError_DmaCaptureFailure_c;
    }
}

/* function to avoid high CCM metric */
static void PDE_TPM_IrqHandler_FirstTime(void)
{
    /* Lock PLL */
    PDE_TsmOverridePll();
    
    /* Lock RX */
    PDE_TsmOverrideRx();
    
    if(gPdeCurrentDeviceType == kPdeDeviceType_RD_c){
        /* Lock TX */
        PDE_TsmOverrideTx();
    }
    
    /* Abort first GENFSK transmission, it is CW from here */
    GENFSK_AbortAll();
    
    /* Turn off the modulator to prevent offset between TX and RX even in CW */
    XCVR_PLL_DIG->MOD_CTRL |= XCVR_PLL_DIG_MOD_CTRL_MOD_DISABLE_MASK;
    
#if !defined(RADIO_IS_GEN_3P5) || USE_DMA_DECIMATION==0
    /* Change OSR */
    PDE_SetOsr32();
#endif
  
}

/* function to avoid high CCM metric */
static void PDE_TPM_IrqHandler_LastTime(void)
{
    //Stop Frequency Sweep algorithm
    PDE_FrequencySweepStop();
    
    //Release TSM Lock
    PDE_TsmOverrideRestore();
    
#if !defined(RADIO_IS_GEN_3P5) || USE_DMA_DECIMATION==0
    // Reset OSR
    PDE_ResetOsr();
#endif
    
    //Unlock AGC
    PDE_ClearManualCalibration();
    
    //Turn the modulator back on
    XCVR_PLL_DIG->MOD_CTRL &= ~XCVR_PLL_DIG_MOD_CTRL_MOD_DISABLE_MASK;
    
    /* Check if we got all the required samples */
    if(gSamplesInBuffer == gRequestedSamples){
        //Report capture complete event
        (void)OSA_EventSet(gPdeEventId, PDE_EVT_DMA_CAPTURE_COMPLETE);
    }
    else{
        //Report capture error
        gPdeReportedError = kPdeError_DmaCaptureFailure_c;
        (void)OSA_EventSet(gPdeEventId, PDE_EVT_INTERNAL_ERROR);
    }
}

static void PDE_TPM_IRQHandler (void){
    static pde_device_sequence_t sequenceState = kPdeDevSeq_RoleSwapOnly_c;
    static uint8_t freqIndex = 0;
    static bool_t firstExecution = true;
    static bool_t mdExtraSwapNeeded = false;
    uint32_t addCarrierOffset = 0;

    DBG_TIMING_TPM_IRQ_HANDLER_START();
    
#if PDE_DEBUG_IQ_CAPTURE_TIMING
    D2_SET();   //Total IRQ Time
#endif  //PDE_DEBUG_IQ_CAPTURE_TIMING
    
    /* Clear TPM Flag */
    TPM_ClearStatusFlags(PDE_TIMER, (uint32_t)kTPM_TimeOverflowFlag);

    /* Update frequency sweep time the first time */
    if(firstExecution){
        /* Update the timer period to the regular switching time */
        TPM_SetTimerPeriod(PDE_TIMER, gNextFsweepCounts);
        PDE_TIMER->CNT = 0;
        
#if defined(RADIO_IS_GEN_3P5)
        /* Disconnect RADIO_TOF_TS_TRIG signal to TPM2_CH0 (set previously as source) */
        SIM->SOPT4 &= ~SIM_SOPT4_TPM2CH0SRC_MASK;
#endif

        if(gPdeCurrentDeviceType == kPdeDeviceType_MD_c){
            /* MD is the first device sampling, so it reaches the requested samples number before the RD.
            On completion, it needs to do an extra TX so RD can capture its last IQ samples */
            mdExtraSwapNeeded = true;
        }
        
        PDE_TPM_IrqHandler_FirstTime();
    }
    
    /* Check if we have all the samples we need */
    if( (gPdeRemainingDmaCaptures != 0U) || mdExtraSwapNeeded){
        pde_freq_calibration_t *freqCal = NULL;
        uint8_t coarse;
        
        if(gPdeRemainingDmaCaptures == 0U){
            mdExtraSwapNeeded = false;
        }
        
        /* Change role in global variable. Radio will be warmed up later */
        if(gPdeCurrentRole == kPdeRole_Rx_c){
            gPdeCurrentRole = kPdeRole_Tx_c;
        }
        else
        {
            /* Debug: Assert if the number of samples already reached the MAX allowed samples */
            assert(gSamplesInBuffer <= PDE_MAX_IQ_SAMPLES);
            gPdeCurrentRole = kPdeRole_Rx_c;
        }
        
        switch (sequenceState) {
          case kPdeDevSeq_RoleSwapOnly_c:
            /* For the first swap, we need to set frequency.
            * If PDE_DEBUG_ADD_CARRIER_OFFSET, update frequency during each swap (since Rx and Tx freq differ)
            */
#if PDE_DEBUG_ADD_CARRIER_OFFSET
            {
#else
            if(firstExecution) {
#endif
              addCarrierOffset = (gPdeCurrentRole == kPdeRole_Tx_c) ? 1U:0U;
                coarse = gPdeCoarseTuneConfigTable[freqIndex];
                freqCal = &gPdeFineTuneConfigTable[addCarrierOffset][freqIndex];
            }
            break;
          case kPdeDevSeq_RoleSwapFreqUpdate_c:
#if PDE_DEBUG_ADD_CARRIER_OFFSET
            addCarrierOffset = (gPdeCurrentRole == kPdeRole_Tx_c) ? 1U:0U;
#endif
            freqIndex++;
            if(freqIndex > gPdeSysCfg.freqSweepNb){
                gPdeReportedError = kPdeError_MaxFrequencyIndexReached_c;
                (void)OSA_EventSet(gPdeEventId, PDE_EVT_INTERNAL_ERROR);
            } else {
                coarse = gPdeCoarseTuneConfigTable[freqIndex];
                freqCal = &gPdeFineTuneConfigTable[addCarrierOffset][freqIndex];
            }
            break;
          /* LCOV_EXCL_START */
          default:
            /* Should not occur */
            break;
          /* LCOV_EXCL_STOP */
        }
        
        /* Perform warmdown/warmup sequence with freq update in between if needed */
        if (gPdeCurrentRole == kPdeRole_Rx_c) {
            /* freqCal being set means we need frequency update */
            if (freqCal != NULL) {
                PDE_TsmSwapRoleToRxWithFreq(coarse, freqCal);
            } else {
                PDE_TsmSwapRoleToRx();
            }
        } else {
            if (freqCal != NULL) {
                PDE_TsmSwapRoleToTxWithFreq(coarse, freqCal);
            } else {
                PDE_TsmSwapRoleToTx();
            }
        }
        
        if(gPdeCurrentRole == kPdeRole_Rx_c)
        {
            /* Wait 10uS for the receiver to stabilize */
#if !defined(RADIO_IS_GEN_3P5)
            PDE_TimeDelayUs(10);
#else
            PDE_TimeDelayUs(100); // todo-optimize
#endif
            if(firstExecution){
#if PDE_DEBUG_DC_RD
                PDE_IqDmaStartCapture(PDE_MAX_NUMBER_OF_TONE_IQ_SAMPLES, &gIqSamplesBuffer32[0], kPdeDmaCapture_Normal_c);
#elif PDE_DEBUG_DC_MD
                asm("BKPT 0");
#else
                /* Start Tone IQ DMA capture */
                PDE_IqDmaStartCapture(PDE_MAX_NUMBER_OF_TONE_IQ_SAMPLES, &gIqSamplesBuffer32[0], kPdeDmaCapture_Normal_c);
#endif
            }
            else{
                /* Start Phase IQ DMA capture */
                PDE_IqDmaStartCapture(gPdeSysCfg.nbOfSamplesPerFrequency, &gIqSamplesBuffer32[gSamplesInBuffer], kPdeDmaCapture_Normal_c);
            }
            
        }
        
        if (sequenceState == kPdeDevSeq_RoleSwapOnly_c) {
            sequenceState = kPdeDevSeq_RoleSwapFreqUpdate_c;
        }
        else {
            sequenceState = kPdeDevSeq_RoleSwapOnly_c;
        }
        
        firstExecution = false;
        
    }
    else{
      
        //Restart first execution Flags
        firstExecution = true;
        sequenceState = kPdeDevSeq_RoleSwapOnly_c;
        freqIndex = 0;
      
        PDE_TPM_IrqHandler_LastTime();
    }
#if PDE_DEBUG_IQ_CAPTURE_TIMING
    D3_CLEAR(); //End of IRQ sequence
#endif  //PDE_DEBUG_IQ_CAPTURE_TIMING

    DBG_TIMING_TPM_IRQ_HANDLER_STOP();
}

#if !defined(RADIO_IS_GEN_3P5)
/* function used only for KW36 */
void PORTB_PORTC_IRQHandler (void){
    uint32_t portbFlags = PORT_GetPinsInterruptFlags(PORTB);

    if( (portbFlags & (1U<<2U)) != 0U ){
        /* Disable interrupt after first trigger */
        PORT_SetPinInterruptConfig(PORTB, 2U, kPORT_InterruptOrDMADisabled);
    }

    PORT_ClearPinsInterruptFlags(PORTB, 0xFFFFFFFFU);
}
#endif
